package com.intellij.css.util;

import com.intellij.openapi.util.Condition;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.css.CssBlock;
import com.intellij.psi.css.CssDeclaration;
import com.intellij.psi.css.CssElement;
import com.intellij.psi.css.CssUri;
import com.intellij.psi.css.resolve.CssResolveManager;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.util.Function;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.io.URLUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.imageio.ImageIO;
import java.util.*;

public final class CssImageUtil {
  private static final String BACKGROUND_IMAGE_PROPERTY_NAME = "background-image";
  private static final String BACKGROUND_PROPERTY_NAME = "background";
  private static final List<String> supportedExtensions = Arrays.asList(ImageIO.getReaderFormatNames());

  private CssImageUtil() {
  }

  /**
   * Retrieves uri elements of background image if given element (or its declaration, or its block) has it.
   *
   * @param element child of background image declaration or background image declaration itself
   * @return psi element that represented uri of image
   */
  @NotNull
  public static Collection<CssUri> findBackgroundImageUris(@NotNull PsiElement element) {
    CssDeclaration declaration = PsiTreeUtil.getNonStrictParentOfType(element, CssDeclaration.class);
    if (declaration != null && isBackgroundDeclaration(declaration)) {
      final Collection<CssUri> uris = PsiTreeUtil.findChildrenOfType(declaration.getValue(), CssUri.class);
      if (!uris.isEmpty()) {
        return uris;
      }
    }

    return Collections.emptyList();
  }

  @NotNull
  public static Collection<CssUri> findBackgroundImageUrlsInBlock(@NotNull CssBlock block) {
    final ArrayList<CssUri> uris = ContainerUtil.newArrayList();
    for (CssDeclaration cssDeclaration : block.getDeclarations()) {
      if (isBackgroundDeclaration(cssDeclaration)) {
        uris.addAll(PsiTreeUtil.findChildrenOfType(cssDeclaration.getValue(), CssUri.class));
      }
    }
    return uris;
  }

  public static boolean isBackgroundDeclaration(@NotNull CssDeclaration declaration) {
    final String propertyName = declaration.getPropertyName();
    return BACKGROUND_IMAGE_PROPERTY_NAME.equalsIgnoreCase(propertyName) || BACKGROUND_PROPERTY_NAME.equalsIgnoreCase(propertyName);
  }

  /**
   * Resolves given uri into virtual files and filter it by type.
   *
   * @param cssUriOrString uri or string referenced to image files
   * @return virtual image files that given CssElement resolves to.
   */
  @NotNull
  public static Collection<VirtualFile> getImageFiles(@Nullable CssElement cssUriOrString) {
    if (cssUriOrString == null) {
      return Collections.emptyList();
    }
    final PsiFile[] psiFiles = CssResolveManager.getInstance().resolveFiles(cssUriOrString, cssUriOrString.getContainingFile());
    final Collection<VirtualFile> virtualFiles = ContainerUtil.map(psiFiles, file -> file.getVirtualFile());
    return ContainerUtil.filter(virtualFiles, file -> {
      if (file == null) {
        return false;
      }
      final String extension = file.getExtension();
      return extension != null && supportedExtensions.contains(extension.toLowerCase(Locale.US));
    });
  }

  /**
   * Check if given element is CssUri and it has data: URL value
   *
   * @param uri psi element to check
   * @return is data uri
   */
  public static boolean isDataUri(@Nullable PsiElement uri) {
    if (uri != null && uri instanceof CssUri) {
      return URLUtil.isDataUri(((CssUri)uri).getValue());
    }
    return false;
  }
}
