package com.intellij.psi.css.descriptor;

import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;

import java.util.Comparator;

public enum CssContextType {
  ANY("any", "everywhere", true),
  NONE("none", "root of file", true),
  NONAME("noname", "regular ruleset", true), //in simple rulesets
  PAGE("@page", "@page", false),
  JQUERY("@jquery", "jquery selector", true), //jquery injections $('<css>') or jQuery('<css>')
  KEYFRAMES("@keyframes", "@keyframes", true),
  CHARSET("@charset", "@charset", true),
  MEDIA("@media", "@media", true),
  NAMESPACE("@namespace", "@namespace", true),
  FONTFACE("@font-face", "@font-face", false),
  VIEWPORT("@viewport", "@viewport", false),
  SUPPORTS("@supports", "@supports", false),
  PAGE_MARGIN("page-margin", "page margin rule", false),
  COLOR_PROFILE("@color-profile", "@color-profile", false), 
  COUNTER_STYLE("@counter-style", "@counter-style", false), 
  UNKNOWN("unknown", "unknown", false);
  
  public static final String[] PAGE_MARGIN_RULES = new String[] {
    "@top-left-corner", "@top-left", "@top-center", "@top-right", "@top-right-corner", "@bottom-left-corner", "@bottom-left",
    "@bottom-center", "@bottom-right", "@bottom-right-corner", "@left-top", "@left-middle", "@left-bottom",
    "@right-top", "@right-middle", "@right-bottom"
  };
  public static final Comparator<? super CssContextType> COMPARATOR = new Comparator<CssContextType>() {
    @Override
    public int compare(@NotNull CssContextType o1, @NotNull CssContextType o2) {
      return o2.getName().compareTo(o1.getName());
    }
  };

  private final String myName;
  
  @NotNull
  private final String myPresentationName;

  public static final CssContextType[] EMPTY_ARRAY = new CssContextType[0];
  /**
   * True means that if context type is subset of any, then css element, allowed in ANY is allowed in given context.
   * Otherwise – not.
   */
  private final boolean mySubsetOfAny;

  CssContextType(String name, @NotNull String presentationName, boolean isSubsetOfAny) {
    myName = name;
    myPresentationName = presentationName;
    mySubsetOfAny = isSubsetOfAny;
  }

  @NotNull
  public static CssContextType fromString(@NotNull String s) {
    for (CssContextType type : values()) {
      if (s.equals(type.getName())) {
        return type;
      }
    }
    if (ArrayUtil.contains(s, PAGE_MARGIN_RULES)) {
      return PAGE_MARGIN;
    }
    throw new IllegalArgumentException("Unknown context type '" + s + "'");
  }

  private String getName() {
    return myName;
  }

  public boolean isSubsetOfAny() {
    return mySubsetOfAny;
  }

  @NotNull
  private String getPresentationName() {
    return myPresentationName;
  }

  @NotNull
  @Override
  public String toString() {
    return getPresentationName();
  }
}
