package com.intellij.psi.css.descriptor.value;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiElement;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

public class CssValueMatchData {
  private final boolean myMatched;
  private final CssValueDescriptor myDescriptor;
  private List<PsiElement> myMatchedTerms = ContainerUtil.newArrayList();
  private final List<CssValueMatchData> myChildren;

  public CssValueMatchData(@NotNull CssValueDescriptor descriptor, boolean matched, @NotNull List<CssValueMatchData> children) {
    myDescriptor = descriptor;
    myMatched = matched;
    myChildren = ContainerUtil.filter(children, Condition.NOT_NULL);
  }

  public CssValueMatchData(@NotNull CssValueDescriptor descriptor, boolean matched) {
    this(descriptor, matched, Collections.<CssValueMatchData>emptyList());
  }

  public static CssValueMatchData fail(@NotNull CssValueDescriptor descriptor, @NotNull List<CssValueMatchData> childrenMatchData) {
    return new CssValueMatchData(descriptor, false, childrenMatchData);
  }

  public static CssValueMatchData fail(@NotNull CssValueDescriptor descriptor) {
    return fail(descriptor, Collections.<CssValueMatchData>emptyList());
  }

  public static CssValueMatchData success(@NotNull CssValueDescriptor descriptor, @NotNull List<CssValueMatchData> childrenMatchData) {
    return new CssValueMatchData(descriptor, true, childrenMatchData);
  }

  public static CssValueMatchData success(@NotNull CssValueDescriptor descriptor) {
    return success(descriptor, Collections.<CssValueMatchData>emptyList());
  }

  public void addMatchedElement(@NotNull PsiElement element) {
    myMatchedTerms.add(element);
  }

  @Nullable
  public PsiElement getFirstMatchedElement() {
    PsiElement item = ContainerUtil.getFirstItem(myMatchedTerms);
    if (item == null) {
      for (CssValueMatchData child : myChildren) {
        PsiElement childCandidate = child.getFirstMatchedElement();
        if (childCandidate != null) {
          if (item == null || childCandidate.getTextRange().getEndOffset() > item.getTextRange().getEndOffset()) {
            item = childCandidate;
          }
        }
      }
    }
    return item;
  }

  @Nullable
  public PsiElement getLastMatchedElement() {
    PsiElement item = ContainerUtil.getLastItem(myMatchedTerms);
    if (item == null) {
      for (CssValueMatchData child : myChildren) {
        PsiElement childCandidate = child.getLastMatchedElement();
        if (childCandidate != null) {
          if (item == null || childCandidate.getTextRange().getEndOffset() > item.getTextRange().getEndOffset()) {
            item = childCandidate;
          }
        }
      }
    }
    return item;
  }

  @NotNull
  public CssValueDescriptor getDescriptor() {
    return myDescriptor;
  }

  @Nullable
  public CssValueDescriptor findDeepestDescriptorForElement(@NotNull PsiElement element) {
    for (CssValueMatchData child : myChildren) {
      CssValueDescriptor childDescriptor = child.findDeepestDescriptorForElement(element);
      if (childDescriptor != null) {
        return childDescriptor;
      }
    }
    
    if (myMatchedTerms.contains(element)) {
      return myDescriptor;
    }

    return null;
  }
  
  @Nullable
  public CssValueMatchData findDeepestNotMatchedData() {
    if (myMatched) {
      return null;
    }
    CssValueMatchData result = this;
    for (CssValueMatchData child : myChildren) {
      CssValueMatchData notMatchedChild = child.findDeepestNotMatchedData();
      if (notMatchedChild != null) {
        return notMatchedChild;
      }
    }
    return result;
  }

  public boolean isMatched() {
    return myMatched;
  }
}
