package com.intellij.css.util;

import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.lang.css.CSSLanguage;
import com.intellij.lang.css.CssLanguageProperties;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.css.*;
import com.intellij.psi.css.descriptor.CssMediaType;
import com.intellij.psi.html.HtmlTag;
import com.intellij.psi.tree.TokenSet;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public class CssPsiUtil {
  @NonNls private static final String EMPTY_STRING = "";

  private CssPsiUtil() {
  }

  @Nullable
  public static CssDeclaration findDeclaration(@NotNull CssBlock cssBlock, @NotNull String propertyName, boolean searchInShorthand) {
    for (CssDeclaration declaration : cssBlock.getDeclarations()) {
      if (propertyName.equalsIgnoreCase(declaration.getPropertyName())) {
        return declaration;
      }

      if (searchInShorthand && declaration.isShorthandProperty()) {
        final List<String> propertyNames = Arrays.asList(declaration.expandShorthandProperty());
        if (propertyNames.contains(propertyName)) {
          return declaration;
        }
      }
    }

    return null;
  }

  public static PsiElement findToken(@NotNull CssElement cssElement, @NotNull TokenSet tokenSet) {
    final ASTNode childByType = cssElement.getNode().findChildByType(tokenSet);
    if (childByType != null) {
      return childByType.getPsi();
    }
    for (PsiElement element : cssElement.getChildren()) {
      if (element instanceof ASTNode && tokenSet.contains(((ASTNode)element).getElementType())) {
        return element;
      }
      if (element instanceof CssElement) {
        PsiElement token = findToken((CssElement)element, tokenSet);
        if (token != null) {
          return token;
        }
      }
    }
    return null;
  }

  @NotNull
  public static Set<String> getRulesetTags(@NotNull CssRuleset ruleset) {
    Set<String> result = ContainerUtil.set();
    for (CssSelector selector : ruleset.getSelectors()) {
      CssSimpleSelector lastElement = ArrayUtil.getLastElement(selector.getSimpleSelectors());
      if (lastElement != null) {
        result.add(lastElement.getElementName());
      }
    }
    return result;
  }

  public static boolean isInFunction(@Nullable PsiElement element) {
    final CssValueOwner termListOwner = PsiTreeUtil.getParentOfType(element, CssValueOwner.class);
    return termListOwner instanceof CssFunction;
  }

  public static TextRange getValueTextRange(@NotNull CssValueOwner valueOwner) {
    CssTermList termList = valueOwner.getValue();
    return termList != null ? termList.getTextRange() : TextRange.EMPTY_RANGE;
  }

  @Nullable
  public static Language getStylesheetLanguage(@Nullable PsiElement context) {
    if (context == null || !context.isValid()) {
      return null;
    }
    try {
      Language contextLanguage = context.getLanguage();
      if (contextLanguage != Language.ANY && !contextLanguage.isKindOf(CSSLanguage.INSTANCE)) {
        return null;
      }
      PsiFile file = context.getContainingFile();
      if (file instanceof StylesheetFile) {
        return file.getLanguage();
      }

      CssStylesheet stylesheet = PsiTreeUtil.getNonStrictParentOfType(context, CssStylesheet.class);
      if (stylesheet != null) {
        return stylesheet.getLanguage();
      }
      return contextLanguage.isKindOf(CSSLanguage.INSTANCE) ? contextLanguage : null;
    }
    catch (PsiInvalidElementAccessException e) {
      return null;
    }
  }

  @NotNull
  public static String getDeclarationsTerminator(@Nullable Language language) {
    return language instanceof CssLanguageProperties
           ? ((CssLanguageProperties)language).getDeclarationsTerminator()
           : CssConstants.SEMICOLON;
  }

  @NotNull
  public static String getDeclarationsTerminatorFromContext(@Nullable PsiElement context) {
    CssLanguageProperties languageProperties = getLanguagePropertiesFromContext(context);
    return languageProperties != null ? languageProperties.getDeclarationsTerminator() : CssConstants.SEMICOLON;
  }

  public static boolean isIndentBasedCssLanguage(@Nullable Language language) {
    return language instanceof CssLanguageProperties && ((CssLanguageProperties)language).isIndentBased();
  }

  public static boolean isIndentBasedCssLanguage(@Nullable PsiElement context) {
    CssLanguageProperties languageProperties = getLanguagePropertiesFromContext(context);
    return languageProperties != null && languageProperties.isIndentBased();
  }

  @Nullable
  private static CssLanguageProperties getLanguagePropertiesFromContext(@Nullable PsiElement context) {
    final Language stylesheetLanguage = getStylesheetLanguage(context);
    if (stylesheetLanguage instanceof CssLanguageProperties) {
      return ((CssLanguageProperties)stylesheetLanguage);
    }
    return null;
  }

  public static PsiElement getFirstDeepestTerm(@Nullable PsiElement element) {
    if (element instanceof CssTermList) {
      PsiElement term = PsiTreeUtil.getNonStrictParentOfType(PsiTreeUtil.getDeepestFirst(element), CssTerm.class);
      if (term != null) {
        PsiElement parent = term.getParent();
        while (parent instanceof CssOperation) {
          term = parent;
          parent = term.getParent();
        }
      }
      return term;
    }
    else if (element instanceof CssLineNames) {
      return element.getFirstChild();
    }
    return element;
  }

  public static PsiElement getLastDeepestTerm(@Nullable PsiElement element) {
    if (element instanceof CssTermList) {
      PsiElement term = PsiTreeUtil.getNonStrictParentOfType(PsiTreeUtil.getDeepestLast(element), CssTerm.class);
      if (term != null) {
        PsiElement parent = term.getParent();
        while (parent instanceof CssOperation) {
          term = parent;
          parent = term.getParent();
        }
      }
      return term;
    }
    return element;
  }

  @NotNull
  public static String getTokenText(@Nullable PsiElement element) {
    return element == null ? EMPTY_STRING : element.getText();
  }

  public static PsiElement replaceToken(@NotNull PsiElement token, String value) throws IncorrectOperationException {
    return ElementManipulators.getManipulator(token).handleContentChange(token, new TextRange(0, token.getTextLength()), value);
  }

  public static int hashCodeForElement(@Nullable PsiElement element) {
    if (element == null || element instanceof PsiWhiteSpace || element instanceof PsiComment) {
      return 0;
    }
    return new CssHashingVisitor().calculate(element);
  }

  @NotNull
  public static Set<CssMediaType> getAllowedMediaTypesInContext(@Nullable PsiElement context) {
    CssMedia media = PsiTreeUtil.getNonStrictParentOfType(context, CssMedia.class);
    if (media != null) {
      return media.getMediaTypes();
    }
    HtmlTag htmlTag = PsiTreeUtil.getNonStrictParentOfType(context, HtmlTag.class);
    if (htmlTag != null && "style".equals(htmlTag.getName())) {
      String attributeValue = htmlTag.getAttributeValue("media");
      if (StringUtil.isNotEmpty(attributeValue)) {
        CssMedia customMedia = CssElementFactory.getInstance(context.getProject()).createMedia(attributeValue, CSSLanguage.INSTANCE);
        if (customMedia != null) {
          return customMedia.getMediaTypes();
        }
      }
    }
    return ContainerUtil.newHashSet(CssMediaType.ALL);
  }

  public static boolean isCustomVariableReference(@NotNull PsiElement element) {
    PsiElement parent = element.getParent();
    if (parent instanceof CssTerm) {
      PsiElement termList = parent.getParent();
      if (termList instanceof CssTermList) {
        PsiElement function = termList.getParent();
        if (function instanceof CssTermList && function.getChildren().length == 1) { // workaround for sass/scss files
          function = function.getParent();
        }
        if (function instanceof CssFunction) {
          if (CssConstants.VAR_FUNCTION_NAME.equalsIgnoreCase(((CssFunction)function).getName())) {
            CssTerm[] terms = ((CssTermList)termList).getTerms();
            return terms.length == 1 || terms.length == 2 && parent.getPrevSibling() == null;
          }
        }
      }
    }
    return false;
  }

  private static class CssHashingVisitor extends CssElementVisitor {
    private int myResult = 0;

    public int calculate(@NotNull PsiElement element) {
      myResult = 0;
      element.accept(this);
      return myResult;
    }

    @Override
    public void visitCssUri(CssUri uri) {
      myResult += 31 * uri.getNode().getElementType().hashCode() + 31 * uri.getValue().hashCode();
    }

    @Override
    public void visitCssString(CssString _string) {
      myResult += 31 * _string.getNode().getElementType().hashCode() + 31 * _string.getText().hashCode();
    }

    @Override
    public void visitCssTerm(CssTerm _term) {
      myResult += 31 * _term.getTermType().hashCode();
      super.visitCssElement(_term);
    }


    @Override
    public void visitCssDeclaration(CssDeclaration declaration) {
      myResult += declaration.equalityHashCode();
    }

    @Override
    public void visitCssElement(CssElement element) {
      if (element instanceof CssValueOwner) {
        CssTermList value = ((CssValueOwner)element).getValue();
        if (value != null) {
          value.accept(this);
        }
        if (element instanceof CssNamedElement) {
          final String name = ((CssNamedElement)element).getName();
          if (name != null) {
            myResult += 31 * name.toLowerCase(Locale.US).hashCode();
          }
        }
        return;
      }
      super.visitCssElement(element);
    }

    @Override
    public void visitElement(PsiElement element) {
      if (!(element instanceof PsiComment) && !(element instanceof PsiWhiteSpace)) {
        myResult += 31 * element.getNode().getElementType().hashCode();
        if (element.getChildren().length == 0) {
          myResult += 31 * element.getText().toLowerCase(Locale.US).hashCode();
        }
        else {
          element.acceptChildren(this);
        }
      }
      super.visitElement(element);
    }
  }
}
