/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.vfs.pointers.VirtualFilePointer;
import com.intellij.spring.facet.beans.CustomSetting;

import javax.swing.*;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Allows customizing handling for specific filesets in Spring facet.
 *
 * @since 2017.1
 */
public abstract class SpringFileSetEditorCustomization {

  public static final ExtensionPointName<SpringFileSetEditorCustomization> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.fileSetEditorCustomization");

  /**
   * Determines whether this customization applies to the given fileset.
   * <p/>
   * NB: {@code instanceof} cannot be used here, use fileset ID and/or files to determine applicability.
   *
   * @param fileSet Fileset to be processed.
   * @return {@code true} if customization should be used.
   */
  public abstract boolean isApplicable(SpringFileSet fileSet);

  /**
   * Returns custom groups of configuration files.
   * <p/>
   * All files contained in these groups will be removed from any of the default groups.
   *
   * @param fileSet Fileset to be processed.
   * @return List of custom config file groups.
   */
  public List<CustomConfigFileGroup> getCustomConfigFileGroups(SpringFileSet fileSet) {
    return Collections.emptyList();
  }

  /**
   * Returns all custom settings.
   * <p/>
   * Will be enabled in facet editor if at least one fileset matches {@link #isApplicable(SpringFileSet)}.
   *
   * @return Custom facet settings.
   */
  public List<CustomSetting> getCustomSettings() {
    return Collections.emptyList();
  }


  public static class CustomConfigFileGroup {

    private final String myName;
    private final Icon myIcon;
    private final Set<VirtualFilePointer> myFiles;

    public CustomConfigFileGroup(String name, Icon icon, Set<VirtualFilePointer> files) {
      myName = name;
      myIcon = icon;
      myFiles = files;
    }

    public String getName() {
      return myName;
    }

    public Icon getIcon() {
      return myIcon;
    }

    public Set<VirtualFilePointer> getFiles() {
      return myFiles;
    }
  }
}
