/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamElement;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamClassAttributeMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.*;
import com.intellij.semantic.SemKey;
import com.intellij.spring.contexts.model.ComponentScanPackagesModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.utils.SpringCacheUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Inheritors should register with {@link #COMPONENT_SCAN_META_KEY} to get builtin support
 * for component-scan-like annotations (e.g. goto annotated elements, highlighting of unresolved package definitions).
 */
public abstract class SpringComponentScan extends CommonModelElement.PsiBase implements JamElement, SpringBeansPackagesScan {
  protected static final String VALUE_ATTR_NAME = "value";
  protected static final String BASE_PACKAGES_ATTR_NAME = "basePackages";
  protected static final String BASE_PACKAGE_CLASSES_ATTR_NAME = "basePackageClasses";

  public static final SemKey<JamAnnotationMeta> COMPONENT_SCAN_META_KEY = SemKey.createKey("SpringComponentScan");
  public static final SemKey<SpringComponentScan> COMPONENT_SCAN_JAM_KEY = JamService.JAM_ELEMENT_KEY.subKey("SpringComponentScan");

  private final PsiClass myPsiElement;


  public SpringComponentScan(@NotNull PsiClass psiElement) {
    myPsiElement = psiElement;
  }

  @NotNull
  @Override
  public final Set<CommonSpringBean> getScannedElements(@NotNull Module module) {
    return SpringCacheUtils.getCreatedCachedBeans(module, this, this::getScannedBeans);
  }

  @NotNull
  protected Set<CommonSpringBean> getScannedBeans(@NotNull Module module) {
    final Collection<SpringBeanPointer> scannedComponents =
      ComponentScanPackagesModel.getScannedComponents(getPsiPackages(),
                                                      module,
                                                      null,
                                                      useDefaultFilters(),
                                                      getExcludeContextFilters(),
                                                      getIncludeContextFilters());
    return ContainerUtil.map2LinkedSet(scannedComponents, SpringBeanPointer.TO_BEAN);
  }

  @NotNull
  protected abstract PsiElementRef<PsiAnnotation> getAnnotationRef();

  @Nullable
  public PsiAnnotation getAnnotation() {return getAnnotationRef().getPsiElement();}

  @NotNull
  @Override
  public PsiClass getPsiElement() {
    return myPsiElement;
  }

  @NotNull
  public Set<PsiPackage> getPsiPackages() {
    Set<PsiPackage> allPackages = ContainerUtil.newLinkedHashSet();

    for (JamStringAttributeMeta.Collection<PsiPackage> packageMeta : getPackageJamAttributes()) {
      addPackages(allPackages, packageMeta);
    }

    addBasePackageClasses(allPackages);
    if (allPackages.isEmpty()) {
      PsiFile file = getPsiElement().getContainingFile();
      if (file instanceof PsiClassOwner) {
        final String packageName = ((PsiClassOwner)file).getPackageName();
        final PsiManager manager = getPsiElement().getManager();
        final PsiPackage psiPackage = JavaPsiFacade.getInstance(manager.getProject()).findPackage(packageName);
        if (psiPackage != null) {
          return Collections.singleton(psiPackage);
        }
      }
    }
    return allPackages;
  }

  private void addBasePackageClasses(Set<PsiPackage> importedResources) {
    final List<JamClassAttributeElement> basePackageClasses = getBasePackageClassAttribute();
    for (JamClassAttributeElement bpc : basePackageClasses) {
      PsiClass psiClass = bpc.getValue();
      if (psiClass != null) {
        final PsiDirectory containingDirectory = psiClass.getContainingFile().getContainingDirectory();

        if (containingDirectory != null) {
          final PsiPackage psiPackage = JavaDirectoryService.getInstance().getPackage(containingDirectory);
          if (psiPackage != null) {
            importedResources.add(psiPackage);
          }
        }
      }
    }
  }

  @NotNull
  protected List<JamClassAttributeElement> getBasePackageClassAttribute() {
    final JamClassAttributeMeta.Collection meta = getBasePackageClassMeta();

    return meta != null ? meta.getJam(getAnnotationRef()) : Collections.emptyList();
  }

  @Nullable
  protected abstract JamClassAttributeMeta.Collection getBasePackageClassMeta();

  private void addPackages(@NotNull Set<PsiPackage> importedResources,
                           @NotNull JamStringAttributeMeta.Collection<PsiPackage> attrMeta) {
    for (JamStringAttributeElement<PsiPackage> element : attrMeta.getJam(getAnnotationRef())) {
      if (element != null) {
        //noinspection ConstantConditions
        ContainerUtil.addIfNotNull(importedResources, element.getValue());
      }
    }
  }

  @NotNull
  protected abstract JamAnnotationMeta getAnnotationMeta();

  @NotNull
  public abstract List<JamStringAttributeMeta.Collection<PsiPackage>> getPackageJamAttributes();

  public boolean processPsiPackages(@NotNull Processor<Pair<PsiPackage, ? extends PsiElement>> processor) {
    List<JamStringAttributeMeta.Collection<PsiPackage>> packageJamAttributes = getPackageJamAttributes();
    List<JamClassAttributeElement> basePackageClasses = getBasePackageClassAttribute();

    int packageDefiningElementsCount = basePackageClasses.size();
    for (JamStringAttributeMeta.Collection<PsiPackage> packageMeta : packageJamAttributes) {
      packageDefiningElementsCount += getAnnotationMeta().getAttribute(getPsiElement(), packageMeta).size();
    }

    boolean useCurrentPackageForScan = packageDefiningElementsCount == 0;
    boolean useAnnotationAsElement = packageDefiningElementsCount == 1;
    PsiElement annotationElement = getAnnotationMeta().getAnnotation(getPsiElement());

    for (JamStringAttributeMeta.Collection<PsiPackage> packageMeta : packageJamAttributes) {
      for (JamStringAttributeElement<PsiPackage> element : getAnnotationMeta().getAttribute(getPsiElement(), packageMeta)) {
        if (element != null) {
          PsiPackage psiPackage = element.getValue();
          if (psiPackage != null) {
            final PsiElement identifyingElement =
              useAnnotationAsElement && annotationElement != null ? annotationElement : element.getPsiElement();
            if (!processor.process(Pair.create(psiPackage, identifyingElement == null ? psiPackage : identifyingElement))) {
              return false;
            }
          }
        }
      }
    }

    for (JamClassAttributeElement bpc : basePackageClasses) {
      PsiClass psiClass = bpc.getValue();
      if (psiClass != null) {
        final PsiDirectory containingDirectory = psiClass.getContainingFile().getContainingDirectory();

        if (containingDirectory != null) {
          final PsiPackage psiPackage = JavaDirectoryService.getInstance().getPackage(containingDirectory);
          if (psiPackage != null) {
            if (!processor.process(Pair.create(psiPackage, useAnnotationAsElement ? annotationElement : bpc.getPsiElement()))) {
              return false;
            }
          }
        }
      }
    }

    if (useCurrentPackageForScan) {
      PsiFile file = getPsiElement().getContainingFile();
      if (file instanceof PsiClassOwner) {
        final String packageName = ((PsiClassOwner)file).getPackageName();
        final PsiManager manager = getPsiElement().getManager();
        final PsiPackage psiPackage = JavaPsiFacade.getInstance(manager.getProject()).findPackage(packageName);
        if (psiPackage != null) {
          if (!processor.process(Pair.create(psiPackage, annotationElement))) {
            return false;
          }
        }
      }
    }

    return true;
  }

  @Override
  public PsiElement getIdentifyingPsiElement() {
    return getAnnotation();
  }
}
