/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.converters.PackageJamConverter;
import com.intellij.spring.model.jam.utils.filters.*;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.List;
import java.util.Set;

public class SpringJamComponentScan extends SpringComponentScan {

  protected static final JamStringAttributeMeta.Collection<PsiPackage> VALUE_ATTRIBUTE_META =
    new JamStringAttributeMeta.Collection<>(VALUE_ATTR_NAME, new PackageJamConverter());

  protected static final JamStringAttributeMeta.Collection<PsiPackage> BASE_PACKAGE_ATTR_META =
    new JamStringAttributeMeta.Collection<>(BASE_PACKAGES_ATTR_NAME, new PackageJamConverter());

  private static final JamClassAttributeMeta.Collection BASE_PACKAGE_CLASS_ATTR_META =
    JamAttributeMeta.classCollection(BASE_PACKAGE_CLASSES_ATTR_NAME);

  protected static final JamBooleanAttributeMeta USE_DEFAULT_FILTERS_META =
    JamAttributeMeta.singleBoolean("useDefaultFilters", true);

  protected static final JamAttributeMeta<List<SpringComponentScanFilter>> INCLUDE_FILTERS_ATTR_META =
    JamAttributeMeta.annoCollection("includeFilters", SpringComponentScanFilter.ANNOTATION_META, SpringComponentScanFilter.class);

  protected static final JamAttributeMeta<List<SpringComponentScanFilter>> EXCLUDE_FILTERS_ATTR_META =
    JamAttributeMeta.annoCollection("excludeFilters", SpringComponentScanFilter.ANNOTATION_META, SpringComponentScanFilter.class);

  protected static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype()
    .addAttribute(VALUE_ATTRIBUTE_META)
    .addAttribute(BASE_PACKAGE_ATTR_META)
    .addAttribute(USE_DEFAULT_FILTERS_META)
    .addAttribute(INCLUDE_FILTERS_ATTR_META)
    .addAttribute(EXCLUDE_FILTERS_ATTR_META);

  public static final SemKey<JamAnnotationMeta> META_KEY = COMPONENT_SCAN_META_KEY.subKey("SpringJamComponentScan");
  private static final SemKey<SpringJamComponentScan> JAM_KEY = COMPONENT_SCAN_JAM_KEY.subKey("SpringJamComponentScan");

  public static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.COMPONENT_SCAN, ARCHETYPE, META_KEY);

  public static final JamClassMeta<SpringJamComponentScan> META =
    new JamClassMeta<>(null, SpringJamComponentScan.class, JAM_KEY).addAnnotation(ANNOTATION_META);

  private final PsiElementRef<PsiAnnotation> myAnnotation;

  public SpringJamComponentScan(@NotNull PsiClass psiElement) {
    super(psiElement);
    myAnnotation = getAnnotationMeta().getAnnotationRef(psiElement);
  }

  public SpringJamComponentScan(PsiAnnotation annotation) {
    super(PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true));
    myAnnotation = PsiElementRef.real(annotation);
  }

  @NotNull
  protected JamAnnotationMeta getAnnotationMeta() {
    return ANNOTATION_META;
  }


  protected JamClassAttributeMeta.Collection getBasePackageClassMeta() {
    return BASE_PACKAGE_CLASS_ATTR_META;
  }

  /**
   * Returns all attributes containing package definitions to scan.
   */
  @NotNull
  public List<JamStringAttributeMeta.Collection<PsiPackage>> getPackageJamAttributes() {
    return ContainerUtil.immutableList(VALUE_ATTRIBUTE_META, BASE_PACKAGE_ATTR_META);
  }

  @Override
  public boolean useDefaultFilters() {
    return getAnnotationMeta().getAttribute(getPsiElement(), USE_DEFAULT_FILTERS_META).getValue();
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Exclude> getExcludeContextFilters() {
    Set<SpringContextFilter.Exclude> excludes = ContainerUtil.newLinkedHashSet();
    for (SpringComponentScanFilter filter : getAnnotationMeta().getAttribute(getPsiElement(), EXCLUDE_FILTERS_ATTR_META)) {
      final FilterType value = filter.getFilterType();
      final Set<PsiClass> classes = filter.getFilteredClasses();
      if (value == FilterType.ASSIGNABLE_TYPE) {
        excludes.add(new ExcludeAssignableFilter(classes));
      }
      else if (value == FilterType.ANNOTATION) {
        excludes.add(new ExcludeAnnotationsFilter(classes));
      }
    }

    return excludes;
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Include> getIncludeContextFilters() {
    Set<SpringContextFilter.Include> includes = ContainerUtil.newLinkedHashSet();
    for (SpringComponentScanFilter filter : getAnnotationMeta().getAttribute(getPsiElement(), INCLUDE_FILTERS_ATTR_META)) {
      final FilterType value = filter.getFilterType();
      final Set<PsiClass> classes = filter.getFilteredClasses();
      if (value == FilterType.ASSIGNABLE_TYPE) {
        includes.add(new IncludeAssignableFilter(classes));
      }
      else if (value == FilterType.ANNOTATION) {
        includes.add(new IncludeAnnotationsFilter(classes));
      }
      else if (value == FilterType.CUSTOM) {
        includes.add(new IncludeCustomFilter(classes));
      }
    }

    return includes;
  }


  @NotNull
  public PsiElementRef<PsiAnnotation> getAnnotationRef() {
    return  myAnnotation;
  }
}
