/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts.jdbc;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.*;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.testContexts.converters.PsiFileResourcePathReferenceConverter;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Set;

/**
 * @since 2017.1
 */
public abstract class SpringTestingSql implements TestingSql {
  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = JAM_ANNOTATION_KEY.subKey("SpringTestingSql");
  private static final SemKey<SpringTestingSql> JAM_KEY = TestingSql.JAM_KEY.subKey("SpringTestingSql");

  public static final JamClassMeta<SpringTestingSql> CLASS_META = new JamClassMeta<>(null, SpringTestingSql.class, JAM_KEY);
  public static final JamMethodMeta<SpringTestingSql> METHOD_META = new JamMethodMeta<>(null, SpringTestingSql.class, JAM_KEY);
  private static final JamAnnotationAttributeMeta.Single<SpringTestingSqlConfig> SQL_CONFIG_ATTR =
    JamAttributeMeta.singleAnno("config", SpringTestingSqlConfig.ANNO_META, SpringTestingSqlConfig.class);


  public static final JamStringAttributeMeta.Collection<List<PsiFile>> SCRIPTS_ATTR_META =
    JamAttributeMeta.collectionString(SCRIPTS_ATTR_NAME, new PsiFileResourcePathReferenceConverter());

  public static final JamStringAttributeMeta.Collection<List<PsiFile>> VALUE_ATTR_META =
    JamAttributeMeta.collectionString(VALUE_ATTR_NAME, new PsiFileResourcePathReferenceConverter());

  public static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype()
    .addAttribute(SCRIPTS_ATTR_META)
    .addAttribute(VALUE_ATTR_META)
    .addAttribute(SQL_CONFIG_ATTR);

  public static final JamAnnotationMeta ANNO_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.TEST_SQL, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    CLASS_META.addAnnotation(ANNO_META);
    METHOD_META.addAnnotation(ANNO_META);
  }

  private final PsiMember myPsiMember;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;


  @SuppressWarnings("unused")
  public SpringTestingSql(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringTestingSql(PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  public SpringTestingSqlConfig getSqlConfig() {
    PsiAnnotation element = myPsiAnnotation.getPsiElement();
    if (element != null) {
      PsiAnnotationMemberValue config = element.findDeclaredAttributeValue("config");
      if (config instanceof PsiAnnotation) {
        return SQL_CONFIG_ATTR.getJam(PsiElementRef.real((PsiAnnotation)config));
      }
    }
    return null;
  }

  @Nullable
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @NotNull
  @Override
  public Set<PsiFile> getScripts() {
    Set<PsiFile> locations = ContainerUtil.newLinkedHashSet();
    addFiles(locations, VALUE_ATTR_META.getJam(myPsiAnnotation));
    addFiles(locations, SCRIPTS_ATTR_META.getJam(myPsiAnnotation));
    return locations;
  }

  private static void addFiles(@NotNull Set<PsiFile> locations, @NotNull List<JamStringAttributeElement<List<PsiFile>>> jam) {
    for (JamStringAttributeElement<List<PsiFile>> element : jam) {
      List<PsiFile> value = element.getValue();
      if (value != null) {
        ContainerUtil.addAllNotNull(locations, value);
      }
    }
  }
}
