/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts.jdbc;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamClassMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.converters.SpringBeanReferenceJamConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static com.intellij.spring.constants.SpringConstants.JAVAX_SQL_DATA_SOURCE;

/**
 * @since 2017.1
 */
public abstract class SpringTestingSqlConfig implements JamElement {
  public static final JamClassMeta<SpringTestingSqlConfig> META = new JamClassMeta<>(SpringTestingSqlConfig.class);
  public static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.TEST_SQL_CONFIG);

  private static final JamStringAttributeMeta.Single<SpringBeanPointer> TRANSACTION_MANAGER_ATTR_META =
    JamAttributeMeta
      .singleString("transactionManager", new SpringBeanReferenceJamConverter(SpringAnnotationsConstants.PLATFORM_TRANSACTION_MANAGER));

  private static final JamStringAttributeMeta.Single<SpringBeanPointer> DATASOURCE_ATTR_META =
    JamAttributeMeta.singleString("dataSource", new SpringBeanReferenceJamConverter(JAVAX_SQL_DATA_SOURCE));

  static {
    META.addAnnotation(ANNO_META);
    ANNO_META.addAttribute(TRANSACTION_MANAGER_ATTR_META).addAttribute(DATASOURCE_ATTR_META);
  }

  private final PsiClass myPsiClass;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  @SuppressWarnings("unused") public SpringTestingSqlConfig(@NotNull PsiClass psiClass) {
    myPsiClass = psiClass;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiClass);
  }

  @SuppressWarnings("unused")
  public SpringTestingSqlConfig(PsiAnnotation annotation) {
    myPsiClass = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @NotNull
  public PsiClass getPsiElement() {return myPsiClass;};

  public boolean isPsiValid() { return getPsiElement().isValid();}

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  public JamStringAttributeElement<SpringBeanPointer> getTransactionManagerElement() {
    return TRANSACTION_MANAGER_ATTR_META.getJam(myPsiAnnotation);
  }

  public JamStringAttributeElement<SpringBeanPointer> getDatasourceAttrElement() {
    return DATASOURCE_ATTR_META.getJam(myPsiAnnotation);
  }
}
