/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.openapi.util.TextRange;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.Stack;

/**
 * @author Yann C&eacute;bron
 */
public class PlaceholderTextRanges {

  private PlaceholderTextRanges() {
  }

  @NotNull
  public static Set<TextRange> getPlaceholderRanges(@NotNull String s, @NotNull String prefix, @NotNull String suffix) {
    return getPlaceholderRanges(s, prefix, suffix, false);
  }

  @NotNull
  public static Set<TextRange> getPlaceholderRanges(@NotNull String s,
                                                    @NotNull String prefix,
                                                    @NotNull String suffix,
                                                    boolean useFullTextRange) {
    return getPlaceholderRanges(s, prefix, suffix, useFullTextRange, false);
  }

  /**
   * @param s                  String to parse.
   * @param prefix             Prefix.
   * @param suffix             Suffix.
   * @param useFullTextRange   Use full text range (incl. prefix/suffix).
   * @param filterNestedRanges Remove nested ranges from result.
   * @return Matching ranges.
   */
  @NotNull
  public static Set<TextRange> getPlaceholderRanges(@NotNull String s,
                                                    @NotNull String prefix,
                                                    @NotNull String suffix,
                                                    boolean useFullTextRange,
                                                    boolean filterNestedRanges) {
    int current = s.indexOf(prefix);
    if (current == -1) {
      return Collections.emptySet();
    }

    final Set<TextRange> ranges = new LinkedHashSet<>(2);

    final Stack<Integer> prefixes = new Stack<>();
    prefixes.push(current);

    while (current >= 0) {
      int nextSuffix = s.indexOf(suffix, current);
      if (nextSuffix <= 0) {
        break;
      }

      int nextPrefix = s.indexOf(prefix, current + 1);

      while (nextPrefix > 0 && nextPrefix < nextSuffix) {
        prefixes.push(nextPrefix);
        nextPrefix = s.indexOf(prefix, nextPrefix + 1);
      }

      int startOffset = prefixes.pop() + (useFullTextRange ? 0 : prefix.length());
      int endOffset = useFullTextRange ? nextSuffix + suffix.length() : nextSuffix;

      final TextRange textRange = new TextRange(startOffset, endOffset);
      ranges.add(textRange);

      if (prefixes.isEmpty()) {
        current = s.indexOf(prefix, nextSuffix + suffix.length());
        prefixes.push(current);
      }
      else {
        current = s.indexOf(prefix, nextSuffix);
        if (current > 0) {
          prefixes.push(current);
        }
        else {
          current = s.indexOf(suffix, nextSuffix + suffix.length());
        }
      }
    }

    return filterNestedRanges ? filterNested(ranges) : ranges;
  }

  private static Set<TextRange> filterNested(Set<TextRange> allRanges) {
    Set<TextRange> filtered = new LinkedHashSet<>(allRanges.size());
    for (TextRange outer : allRanges) {
      boolean contains = anyRangeContains(allRanges, outer);
      if (!contains) filtered.add(outer);
    }

    return filtered;
  }

  private static boolean anyRangeContains(Set<TextRange> allRanges, TextRange inner) {
    for (TextRange outer : allRanges) {
      if (!inner.equals(outer) &&
          outer.contains(inner)) {
        return true;
      }
    }
    return false;
  }
}
