/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.xml;

import com.intellij.pom.references.PomService;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.DefaultSpringBeanQualifier;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.xml.DomReflectionUtil;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

public abstract class DomSpringBeanImpl extends AbstractDomSpringBean implements DomSpringBean {

  @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
  private static final Map<Class, BeanName> ourBeanNameCache = new ConcurrentFactoryMap<Class, BeanName>() {
    @Nullable
    @Override
    protected BeanName create(Class key) {
      return DomReflectionUtil.findAnnotationDFS(key, BeanName.class);
    }
  };

  @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
  private static final Map<Class, BeanType> ourBeanTypeCache = new ConcurrentFactoryMap<Class, BeanType>() {
    @Nullable
    @Override
    protected BeanType create(Class key) {
      return DomReflectionUtil.findAnnotationDFS(key, BeanType.class);
    }
  };

  /**
   * Use {@link BeanName} to provide custom bean name.
   * Overriding recommended when {@link #setName(String)} is also changed.
   * NB: do not use any resolving when overriding.
   *
   * @return Bean name.
   */
  @SuppressWarnings("unchecked")
  @Nullable
  public String getBeanName() {
    final BeanName beanName = ourBeanNameCache.get(getClass());
    if (beanName == null) {
      return DomUtil.hasXml(getId()) ? getId().getRawText() : null;
    }

    final String value = beanName.value();
    if (!value.isEmpty()) {
      return value;
    }

    // we already check in indexing, but if model version is not incremented such errors might be introduced
    final Class<? extends BeanNameProvider> providerClass = beanName.provider();
    if (providerClass == BeanNameProvider.class) {
      throw new IllegalStateException("@BeanName: no value() given, provider() must be set" + getDomElementType());
    }

    final BeanNameProvider beanNameProvider = BeanProviderRegistry.getInstance().getBeanNameProvider(providerClass);
    return beanNameProvider.getBeanName(this);
  }

  public void setName(@NotNull String newName) {
    if (DomUtil.hasXml(getId())) {
      getId().setStringValue(newName);
    }
  }

  /**
   * @return Bean's class name.
   * @see BeanType
   */
  @SuppressWarnings("unchecked")
  @Nullable
  public final String getClassName() {
    final BeanType beanType = ourBeanTypeCache.get(getClass());
    if (beanType == null) {
      return null;
    }

    final String value = beanType.value();
    if (!value.isEmpty()) {
      return value;
    }

    // we already check in indexing, but if model version is not incremented such errors might be introduced
    final Class<? extends BeanTypeProvider> providerClass = beanType.provider();
    if (providerClass == BeanTypeProvider.class) {
      throw new IllegalStateException("@BeanType: no value() given, provider() must be set" + getDomElementType());
    }

    final BeanTypeProvider beanTypeProvider = BeanProviderRegistry.getInstance().getBeanTypeProvider(providerClass);
    return beanTypeProvider.getBeanType(this);
  }

  @NotNull
  public PsiFile getContainingFile() {
    return DomUtil.getFile(this);
  }

  @Nullable
  public PsiElement getIdentifyingPsiElement() {
    if (!isValid()) return null;
    return PomService.convertToPsi(getManager().getProject(),
                                   BeanService.getInstance().createSpringBeanPsiTarget(this));
  }

  @NotNull
  public String[] getAliases() {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  public PsiManager getPsiManager() {
    return PsiManager.getInstance(getManager().getProject());
  }

  @Override
  public Beans getBeansParent() {
    return DomUtil.getParentOfType(this, Beans.class, false);
  }

  @Override
  public SpringQualifier getSpringQualifier() {
    return DefaultSpringBeanQualifier.create(this);
  }
}
