/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.references;

import com.intellij.codeInsight.ExpectedTypeInfo;
import com.intellij.codeInsight.ExpectedTypesProvider;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SpringBeanNamesReferenceProvider extends PsiReferenceProvider {

  public static final String[] METHODS = new String[]{"containsBean", "getBean", "isSingleton", "getType", "getAliases"};

  @NotNull
  public PsiReference[] getReferencesByElement(@NotNull PsiElement element, @NotNull final ProcessingContext context) {
    if (element instanceof PsiLiteral) {
      final PsiLiteral literalExpression = (PsiLiteral)element;
      if (literalExpression.getValue() instanceof String) {
        return new PsiReference[]{
          new SpringBeanReference(literalExpression,
                                  ElementManipulators.getManipulator(element).getRangeInElement(element),
                                  determineRequiredClass(element),
                                  false)
        };
      }
    }
    return PsiReference.EMPTY_ARRAY;
  }

  /**
   * Tries to determine required bean class by analyzing surrounding expression.
   *
   * @param element Current element.
   * @return Expected bean class or {@code null} if not determinable.
   */
  @Nullable
  public static PsiClass determineRequiredClass(PsiElement element) {
    final PsiExpression expression = PsiTreeUtil.getParentOfType(element, PsiExpression.class);
    if (expression == null) {
      return null;
    }

    final ExpectedTypeInfo[] types = ExpectedTypesProvider.getExpectedTypes(expression, true);
    if (types.length != 1) {
      return null;
    }

    final PsiType type = types[0].getType();
    return PsiTypesUtil.getPsiClass(type);
  }
}
