/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.persistence.util;

import com.intellij.facet.Facet;
import com.intellij.facet.FacetManager;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.persistence.PersistenceHelper;
import com.intellij.persistence.facet.PersistenceFacet;
import com.intellij.persistence.model.*;
import com.intellij.persistence.roles.PersistenceClassRole;
import com.intellij.persistence.roles.PersistenceClassRoleEnum;
import com.intellij.persistence.roles.PersistenceRoleHolder;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.ProjectScope;
import com.intellij.psi.util.*;
import com.intellij.util.*;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericValue;
import gnu.trove.THashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * @author Gregory.Shrago
 */
public class PersistenceCommonUtil {

  private PersistenceCommonUtil() { }

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacets(@NotNull final Project project) {
    List<PersistenceFacet> result = Collections.emptyList();
    for (Module module : ModuleManager.getInstance(project).getModules()) {
      List<PersistenceFacet> facets = getAllPersistenceFacets(module);
      if (!facets.isEmpty()) {
        if (result.isEmpty()) result = new ArrayList<>(facets.size());
        result.addAll(facets);
      }
    }
    return result;
  }

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacets(@NotNull final Module module) {
    List<PersistenceFacet> result = Collections.emptyList();
    for (Facet facet : FacetManager.getInstance(module).getAllFacets()) {
      if (facet instanceof PersistenceFacet) {
        if (result.isEmpty()) result = new ArrayList<>();
        result.add((PersistenceFacet)facet);
      }
    }
    return result;
  }

  private static final Key<CachedValue<List<PersistenceFacet>>> MODULE_PERSISTENCE_FACETS = Key.create("MODULE_PERSISTENCE_FACETS");

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacetsWithDependencies(@NotNull final Module module) {
    if (module.isDisposed()) return Collections.emptyList();
    CachedValue<List<PersistenceFacet>> cachedValue =
      module.getUserData(MODULE_PERSISTENCE_FACETS);
    if (cachedValue == null) {
      cachedValue = CachedValuesManager.getManager(module.getProject()).createCachedValue(() -> {
        final Set<Module> modules = new THashSet<>();
        ContainerUtil.addAll(modules, JamCommonUtil.getAllDependentModules(module));
        ContainerUtil.addAll(modules, JamCommonUtil.getAllModuleDependencies(module));
        final Set<PersistenceFacet> facets =
          new THashSet<>();
        for (Module depModule : modules) {
          facets.addAll(getAllPersistenceFacets(depModule));
        }
        return new CachedValueProvider.Result<List<PersistenceFacet>>(
          new ArrayList<>(facets),
          ProjectRootManager.getInstance(module.getProject()));
      }, false);
      module.putUserData(MODULE_PERSISTENCE_FACETS, cachedValue);
    }
    return cachedValue.getValue();
  }

  public static PersistenceModelBrowser createSameUnitsModelBrowser(@Nullable final PsiElement sourceElement) {
    final PsiClass sourceClass;
    final Set<PersistencePackage> unitSet;
    if (sourceElement == null || (sourceClass = PsiTreeUtil.getParentOfType(sourceElement, PsiClass.class, false)) == null) {
      unitSet = null;
    }
    else {
      unitSet = getAllPersistenceUnits(sourceClass, new THashSet<>());
    }
    return createUnitsAndTypeMapper(unitSet);
  }

  public static PersistenceModelBrowser createSameUnitsModelBrowser(@Nullable final DomElement sourceDom) {
    final Set<PersistencePackage> unitSet;
    final DomElement rootElement;
    if (sourceDom == null || !((rootElement = DomUtil.getFileElement(sourceDom).getRootElement()) instanceof PersistenceMappings)) {
      unitSet = null;
    }
    else {
      unitSet = new THashSet<>(PersistenceHelper.getHelper().getSharedModelBrowser().getPersistenceUnits((PersistenceMappings)rootElement));
    }
    return createUnitsAndTypeMapper(unitSet);
  }

  public static PersistenceModelBrowser createUnitsAndTypeMapper(@Nullable final Set<PersistencePackage> unitSet) {
    return PersistenceHelper.getHelper().createModelBrowser().setRoleFilter(role -> {
      final PersistentObject object = role.getPersistentObject();
      final PersistenceClassRoleEnum roleType = role.getType();
      return roleType != PersistenceClassRoleEnum.ENTITY_LISTENER &&
             object != null &&
             (unitSet == null || unitSet.contains(role.getPersistenceUnit()));
    });
  }

  public static PersistenceModelBrowser createFacetAndUnitModelBrowser(final PersistenceFacet facet,
                                                                       final PersistencePackage unit,
                                                                       @Nullable final PersistenceClassRoleEnum type) {
    return PersistenceHelper.getHelper().createModelBrowser().setRoleFilter(role -> {
      final PersistentObject object = role.getPersistentObject();
      return object != null && (type == null || role.getType() == type) && (unit == null || unit.equals(role.getPersistenceUnit())) &&
             (facet == null || facet.equals(role.getFacet()));
    });
  }

  @Nullable
  public static PsiType getTargetEntityType(final PsiMember psiMember) {
    return getTargetEntityType(PropertyUtil.getPropertyType(psiMember));
  }

  @Nullable
  public static PsiType getTargetEntityType(final PsiType type) {
    return getTypeInfo(type).getValueType();
  }

  public static <T extends Collection<PersistencePackage>> T getAllPersistenceUnits(@Nullable final PsiClass sourceClass,
                                                                                    @NotNull final T result) {
    for (PersistenceClassRole role : getPersistenceRoles(sourceClass)) {
      ContainerUtil.addIfNotNull(result, role.getPersistenceUnit());
    }
    return result;
  }

  @NotNull
  public static <V extends PersistenceMappings> Collection<V> getDomEntityMappings(
    final Class<V> mappingsClass, final PersistencePackage unit, final PersistenceFacet facet) {
    final THashSet<V> result = new THashSet<>();
    for (PersistenceMappings mappings : facet.getDefaultEntityMappings(unit)) {
      if (ReflectionUtil.isAssignable(mappingsClass, mappings.getClass())) {
        result.add((V)mappings);
      }
    }
    for (GenericValue<V> value : unit.getModelHelper().getMappingFiles(mappingsClass)) {
      ContainerUtil.addIfNotNull(result, value.getValue());
    }
    return result;
  }

  public static boolean isSameTable(final TableInfoProvider table1, final TableInfoProvider table2) {
    if (table1 == null || table2 == null) return false;
    final String name1 = table1.getTableName().getValue();
    return StringUtil.isNotEmpty(name1) &&
           Comparing.equal(name1, table2.getTableName().getValue()) &&
           Comparing.equal(table1.getSchema().getValue(), table2.getSchema().getValue()) &&
           Comparing.equal(table1.getCatalog().getValue(), table2.getCatalog().getValue());
  }

  public static String getUniqueId(final PsiElement psiElement) {
    final VirtualFile virtualFile = psiElement == null ? null : PsiUtilBase.getVirtualFile(psiElement);
    return virtualFile == null ? "" : virtualFile.getUrl();
  }

  public static String getMultiplicityString(final boolean optional, final boolean many) {
    final String first = (optional ? "0" : "1");
    final String last = (many ? "*" : "1");
    return first.equals(last) ? first : first + ".." + last;
  }

  public static <T, V extends Collection<T>> V mapPersistenceRoles(final V result,
                                                                   final Project project,
                                                                   final PersistenceFacet facet,
                                                                   final PersistencePackage unit,
                                                                   final Function<PersistenceClassRole, T> mapper) {
    for (PersistenceClassRole role : getPersistenceRoles(project)) {
      if ((facet == null || facet == role.getFacet()) && (unit == null || unit == role.getPersistenceUnit())) {
        ContainerUtil.addIfNotNull(result, mapper.fun(role));
      }
    }
    return result;
  }

  public static boolean haveCorrespondingMultiplicity(final PersistentRelationshipAttribute a1, final PersistentRelationshipAttribute a2) {
    return a1.getAttributeModelHelper().getRelationshipType().corresponds(a2.getAttributeModelHelper().getRelationshipType());
  }


  @NotNull
  public static JavaTypeInfo getTypeInfo(final PsiType type) {
    return getTypeInfo(type, null);
  }

  @NotNull
  public static JavaTypeInfo getTypeInfo(final PsiType type, @Nullable PsiClass convertClass) {
    if (type instanceof PsiArrayType) {
      return new JavaTypeInfo(JavaContainerType.ARRAY, type, convertClass, ((PsiArrayType)type).getComponentType());
    }
    final PsiClassType.ClassResolveResult classResolveResult = type instanceof PsiClassType ? ((PsiClassType)type).resolveGenerics() : null;
    final PsiClass psiClass = classResolveResult == null ? null : classResolveResult.getElement();
    if (psiClass == null) return new JavaTypeInfo(null, type,convertClass);
    final PsiManager manager = psiClass.getManager();
    final GlobalSearchScope scope = ProjectScope.getAllScope(manager.getProject());
    for (JavaContainerType collectionType : JavaContainerType.values()) {
      if (collectionType == JavaContainerType.ARRAY) continue;
      final PsiClass aClass = JavaPsiFacade.getInstance(manager.getProject()).findClass(collectionType.getJavaBaseClassName(), scope);
      if (aClass != null && (manager.areElementsEquivalent(aClass, psiClass) || psiClass.isInheritor(aClass, true))) {
        final PsiSubstitutor superClassSubstitutor =
          TypeConversionUtil.getSuperClassSubstitutor(aClass, psiClass, classResolveResult.getSubstitutor());
        final JavaTypeInfo result = new JavaTypeInfo(collectionType, type, convertClass, ArrayUtil.reverseArray(
          ContainerUtil.map2Array(aClass.getTypeParameters(), PsiType.class,
                                  (NullableFunction<PsiTypeParameter, PsiType>)psiTypeParameter -> superClassSubstitutor
                                    .substitute(psiTypeParameter))));
        if (result.containerType == JavaContainerType.MAP && result.parameters.length != 2
            || JavaContainerType.isCollection(result.containerType) && result.parameters.length != 1) {
          return new JavaTypeInfo(null, type);
        }
        return result;
      }
    }
    return new JavaTypeInfo(null, type, convertClass);
  }

  public static Query<PersistentObject> queryPersistentObjects(final PersistenceMappings mappings) {
    return new ExecutorsQuery<>(mappings, Collections.<QueryExecutor<PersistentObject, PersistenceMappings>>singletonList(
      new QueryExecutor<PersistentObject, PersistenceMappings>() {

        public boolean execute(@NotNull PersistenceMappings queryParameters, @NotNull Processor<PersistentObject> consumer) {
          if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentEntities(), consumer)) return false;
          if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentSuperclasses(), consumer)) return false;
          if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentEmbeddables(), consumer)) return false;
          return true;
        }
      }));
  }

  @Nullable
  public static PsiClass getTargetClass(final PersistentRelationshipAttribute attribute) {
    final GenericValue<PsiClass> classValue = attribute.getTargetEntityClass();
    final PsiClass targetClass;
    if (classValue.getStringValue() != null) {
      targetClass = classValue.getValue();
    }
    else {
      final PsiType entityType = getTargetEntityType(attribute.getPsiMember());
      targetClass = entityType instanceof PsiClassType ? ((PsiClassType)entityType).resolve() : null;
    }
    return targetClass;
  }

  @Nullable
  public static PsiClass getTargetClass(final PersistentEmbeddedAttribute attribute) {
    final GenericValue<PsiClass> classValue = attribute.getTargetEmbeddableClass();
    final PsiClass targetClass;
    if (classValue.getStringValue() != null) {
      targetClass = classValue.getValue();
    }
    else {
      final PsiType entityType = PropertyUtil.getPropertyType(attribute.getPsiMember());
      targetClass = entityType instanceof PsiClassType ? ((PsiClassType)entityType).resolve() : null;
    }
    return targetClass;
  }

  @Nullable
  public static PsiType getPrimaryKeyClass(@NotNull PersistentEntityBase persistentObject, final PersistenceModelBrowser browser) {
    final ArrayList<PersistentAttribute> idAttrs = new ArrayList<>();
    for (PersistentObject object : browser.queryPersistentObjectHierarchy(persistentObject)) {
      if (object instanceof PersistentEntityBase) {
        final PsiClass idClass = ((PersistentEntityBase)object).getIdClassValue().getValue();
        if (idClass != null) return JavaPsiFacade.getElementFactory(idClass.getProject()).createType(idClass);
        for (PersistentAttribute idAttr : object.getObjectModelHelper().getAttributes()) {
          if (idAttr.getAttributeModelHelper().isIdAttribute()) {
            idAttrs.add(idAttr);
          }
        }
      }
    }
    return idAttrs.size() == 1 ? idAttrs.get(0).getPsiType() : null;
  }

  public static PersistenceClassRole[] getPersistenceRoles(final PsiClass psiClass) {
    if (psiClass == null) return PersistenceClassRole.EMPTY_ARRAY;

    CommonProcessors.CollectProcessor<PersistenceClassRole> collectProcessor = new CommonProcessors.CollectProcessor<>();
    PersistenceRoleHolder.getInstance(psiClass.getProject()).processAllRoles(psiClass, collectProcessor);

    return ContainerUtil.toArray(collectProcessor.getResults(), PersistenceClassRole.ARRAY_FACTORY);
  }

  public static PersistenceClassRole[] getPersistenceRoles(@NotNull Project project) {
    CommonProcessors.CollectProcessor<PersistenceClassRole> collectProcessor = new CommonProcessors.CollectProcessor<>();
    PersistenceRoleHolder.getInstance(project).processAllRoles(collectProcessor);

    return ContainerUtil.toArray(collectProcessor.getResults(), PersistenceClassRole.ARRAY_FACTORY);
  }
}
