/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database;

import com.intellij.database.model.DatabaseSystem;
import com.intellij.database.model.RawConnectionConfig;
import com.intellij.openapi.util.Version;
import com.intellij.util.ConcurrencyUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.ConcurrentMap;

/**
 * @author gregsh
 */
public final class DatabaseFamilyId {

  private final static ConcurrentMap<String, DatabaseFamilyId> ourFamilies = ContainerUtil.newConcurrentMap();

  @NotNull
  public static DatabaseFamilyId create(@NotNull String name) {
    DatabaseFamilyId existing = ourFamilies.get(name);
    if (existing != null) return existing;
    return ConcurrencyUtil.cacheOrGet(ourFamilies, name, new DatabaseFamilyId(name));
  }


  // supported dialects
  public static final DatabaseFamilyId UNKNOWN = create("UNKNOWN");
  public static final DatabaseFamilyId ORACLE = create("ORACLE");
  public static final DatabaseFamilyId MYSQL = create("MYSQL");
  public static final DatabaseFamilyId POSTGRES = create("POSTGRES");
  public static final DatabaseFamilyId MSSQL = create("MSSQL");
  public static final DatabaseFamilyId SYBASE = create("SYBASE");
  public static final DatabaseFamilyId DB2 = create("DB2");
  public static final DatabaseFamilyId SQLITE = create("SQLITE");
  public static final DatabaseFamilyId HSQLDB = create("HSQLDB");
  public static final DatabaseFamilyId H2 = create("H2");
  public static final DatabaseFamilyId DERBY = create("DERBY");

  // unsupported dialects
  public static final DatabaseFamilyId MARIADB = create("MARIADB");
  public static final DatabaseFamilyId REDSHIFT = create("REDSHIFT");
  public static final DatabaseFamilyId VERTICA = create("VERTICA");
  public static final DatabaseFamilyId PRESTO = create("PRESTO");
  public static final DatabaseFamilyId INFORMIX = create("INFORMIX");
  public static final DatabaseFamilyId IMPALA = create("IMPALA");
  public static final DatabaseFamilyId NETEZZA = create("NETEZZA");
  public static final DatabaseFamilyId PHOENIX = create("PHOENIX");
  public static final DatabaseFamilyId HIVE = create("HIVE");
  public static final DatabaseFamilyId SNOWFLAKE = create("SNOWFLAKE");
  public static final DatabaseFamilyId INGRES = create("INGRES");
  public static final DatabaseFamilyId TERADATA = create("TERADATA");
  public static final DatabaseFamilyId OPENEDGE = create("OPENEDGE");
  public static final DatabaseFamilyId TIBERO = create("TIBERO");
  public static final DatabaseFamilyId FILEMAKER = create("FILEMAKER");

  private final String myName;

  private DatabaseFamilyId(@NotNull String name) {
    myName = name;
  }

  @NotNull
  public String getName() {
    return myName;
  }

  @Override
  public String toString() {
    return getName();
  }

  public boolean isOracle() { return this == ORACLE; }
  public boolean isMysql() { return this == MYSQL || this == MARIADB; }
  public boolean isPostgres() { return this == POSTGRES || this == REDSHIFT; }
  public boolean isMicrosoft() { return this == MSSQL; }
  public boolean isSybase() { return this == SYBASE; }
  public boolean isDb2() { return this == DB2; }
  public boolean isHsqldb() { return this == HSQLDB; }
  public boolean isH2() { return this == H2; }
  public boolean isDerby() { return this == DERBY; }
  public boolean isSqlite() { return this == SQLITE; }
  public boolean isTransactSql() { return isMicrosoft() || isSybase(); }

  @NotNull
  public static DatabaseFamilyId forDataSource(@NotNull DatabaseSystem o) {
    DatabaseFamilyId result = fromString(o.getDatabaseProductName());
    if (result == UNKNOWN && o instanceof RawConnectionConfig) {
      result = forConnection((RawConnectionConfig)o);
    }
    if (result == POSTGRES) {
      String str = o.getDatabaseProductVersion();
      Version version = str != null && str.contains("8") ? Version.parseVersion(str) : null;
      if (version != null && version.major == 8 && version.bugfix == 2) {
        result = REDSHIFT;
      }
    }
    return result;
  }

  @NotNull
  public static DatabaseFamilyId forConnection(@Nullable RawConnectionConfig o) {
    if (o == null) return UNKNOWN;
    DatabaseFamilyId result = fromString(o.getUrl());
    if (result != UNKNOWN) return result;
    return fromString(o.getDriverClass());
  }

  @NotNull
  public static DatabaseFamilyId fromString(@Nullable String text) {
    if (text == null) return UNKNOWN;
    String pattern = "(?i).*(?:%s).*";
    for (DatabaseFamilyId family : ourFamilies.values()) {
      if (text.matches(String.format(pattern, family.getName()))) return family;
    }
    if (text.matches(String.format(pattern, "hsql"))) return HSQLDB;
    if (text.matches(String.format(pattern, "microsoft|sqlserver"))) return MSSQL;
    if (text.matches(String.format(pattern, "adaptive server")) || text.startsWith("ase")) return SYBASE;
    if (text.startsWith("ids")) return INFORMIX;
    return UNKNOWN;
  }

}
