/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.css;

import com.intellij.lang.Language;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.css.resolve.CssResolver;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

public interface CssSelector extends CssElement {
  enum Combinator {
    DESCENDANT(' '),
    CHILD('>'),
    SIBLING('+'),
    FOLLOWING_SIBLING('~'),
    COLUMN('|', "||");

    private final char myIdentifier;
    private final String myPresentation;

    Combinator(char identifier, String presentation) {
      myIdentifier = identifier;
      myPresentation = presentation;
    }

    Combinator(char identifier) {
      this(identifier, null);
    }

    public char getIdentifier() {
      return myIdentifier;
    }

    public String getPresentation() {
      return StringUtil.notNullize(myPresentation, String.valueOf(myIdentifier));
    }

    public static Combinator fromIdentifier(char c) {
      for (Combinator combinator : values()) {
        if (combinator.myIdentifier == c) {
          return combinator;
        }
      }
      throw new IllegalArgumentException("Illegal combinator identifier: " + c);
    }
  }

  CssSelector[] EMPTY_ARRAY = new CssSelector[0];
  
  @NotNull
  String getPresentableText();

  /**
   * @deprecated breaks stub tree, use {@link this#getCombinators()} and {@link this#getSimpleSelectors()} instead
   * 
   * @return CssSimpleSelectors, XhtmlTokens (CSS_GT ('>'), CSS_PLUS ('+'))
   */
  PsiElement[] getElements();

  @NotNull
  Combinator[] getCombinators();

  @NotNull
  CssSimpleSelector[] getSimpleSelectors();

  Specificity getSpecificity();

  boolean processAmpersandEvaluatedSelectors(@NotNull CssSelector originalSelector,
                                             @Nullable Language language,
                                             @NotNull Processor<CssSelector> processor);

  boolean isAmpersandSelector();

  boolean isMatch(XmlTag tag, CssResolver resolver, @Nullable Map<XmlTag, CssSimpleSelector> mapping);

  @Nullable
  CssRuleset getRuleset();
}
