/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.codeInsight.completion.CompletionUtil;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.DelimitedListProcessor;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.pom.PomTarget;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PsiPackageReference;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.psi.util.PsiUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringManager;
import com.intellij.spring.SpringPresentationProvider;
import com.intellij.spring.contexts.model.SpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.xml.CustomBean;
import com.intellij.spring.model.xml.CustomBeanWrapper;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.RequiredBeanType;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;

public class SpringConverterUtil {

  private SpringConverterUtil() {
  }

  @Nullable
  public static SpringModel getSpringModel(final ConvertContext context) {
    return getSpringModel(context.getInvocationElement());
  }

  @Nullable
  public static SpringModel getSpringModel(final DomElement element) {
    final XmlFile xmlFile = (XmlFile)DomUtil.getFile(element).getOriginalFile();
    return SpringManager.getInstance(xmlFile.getProject()).getSpringModelByFile(xmlFile);
  }

  @Nullable
  public static DomSpringBean getCurrentBean(final ConvertContext context) {
    return getCurrentBean(context.getInvocationElement());
  }

  @Nullable
  public static CommonSpringBean getCurrentBeanCustomAware(final ConvertContext context) {
    DomSpringBean bean = getCurrentBean(context);
    if (bean instanceof CustomBeanWrapper) {
      final CustomBeanWrapper wrapper = (CustomBeanWrapper)bean;
      List<CustomBean> list = wrapper.getCustomBeans();
      if (!list.isEmpty()) {
        return list.get(0);
      }
    }
    return bean;
  }

  @Nullable
  public static DomSpringBean getCurrentBean(final DomElement element) {
    final XmlTag tag = element.getXmlTag();
    if (tag != null) {
      final XmlTag originalElement = CompletionUtil.getOriginalElement(tag);
      if (originalElement != tag) {
        final DomElement domElement = DomManager.getDomManager(tag.getProject()).getDomElement(originalElement);
        if (domElement != null) {
          final DomSpringBean springBean = domElement.getParentOfType(DomSpringBean.class, false);
          if (springBean != null) {
            return springBean;
          }
        }
      }
    }
    return element.getParentOfType(DomSpringBean.class, false);
  }

  @NotNull
  public static Collection<PsiPackage> getPsiPackages(@NotNull PsiReference... psiReferences) {
    final Collection<PsiPackage> list = new LinkedHashSet<>();
    for (PsiReference psiReference : psiReferences) {
      if (psiReference instanceof PsiPackageReference) {
        list.addAll(((PsiPackageReference)psiReference).getReferenceSet().resolvePackage());
      }
    }
    return list;
  }

  @NotNull
  public static List<PsiClassType> getRequiredBeanTypeClasses(ConvertContext context) {
    final DomElement element = context.getInvocationElement();
    final RequiredBeanType type = element.getAnnotation(RequiredBeanType.class);
    if (type == null) {
      return Collections.emptyList();
    }

    List<PsiClassType> types = new SmartList<>();
    for (String className : type.value()) {
      final PsiClass psiClass = DomJavaUtil.findClass(className, element);
      if (psiClass != null) {
        types.add(PsiTypesUtil.getClassType(psiClass));
      }
    }

    return types;
  }

  public static Collection<SpringBeanPointer> getSmartVariants(CommonSpringBean currentBean,
                                                               List<PsiClassType> requiredClasses,
                                                               CommonSpringModel model) {
    final List<SpringBeanPointer> variants = new SmartList<>();
    for (final PsiClassType requiredClass : requiredClasses) {
      final PsiClass psiClass = PsiUtil.resolveClassInType(requiredClass);
      if (psiClass != null) {
        final SpringModelSearchParameters.BeanClass searchParameters =
          SpringModelSearchParameters.byClass(psiClass).withInheritors().effectiveBeanTypes();
        processBeans(model, variants, SpringModelSearchers.findBeans(model, searchParameters),
                     false, currentBean);
      }
      final PsiClass componentClass = PsiUtil.resolveClassInType(PsiUtil.extractIterableTypeParameter(requiredClass, false));
      if (componentClass != null) {
        final SpringModelSearchParameters.BeanClass searchParameters =
          SpringModelSearchParameters.byClass(componentClass).withInheritors().effectiveBeanTypes();
        processBeans(model, variants, SpringModelSearchers.findBeans(model, searchParameters),
                     false, currentBean);
      }
    }
    return variants;
  }

  public static void processBeans(@NotNull CommonSpringModel model,
                                  @NotNull List<SpringBeanPointer> variants,
                                  @NotNull Collection<SpringBeanPointer> pointers,
                                  boolean acceptAbstract,
                                  @Nullable CommonSpringBean currentBean) {
    for (SpringBeanPointer bean : pointers) {
      if (!acceptAbstract && bean.isAbstract()) continue;
      if (bean.isReferenceTo(currentBean)) continue;

      final String beanName = bean.getName();
      if (beanName != null) {
        for (String string : model.getAllBeanNames(beanName)) {
          if (StringUtil.isNotEmpty(string)) {
            variants.add(bean.derive(string));
          }
        }
      }
    }
  }

  @Nullable
  public static LookupElement createCompletionVariant(@NotNull SpringBeanPointer variant) {
    final String name = variant.getName();
    if (name == null) {
      return null;
    }
    return createCompletionVariant(variant, name);
  }

  @Nullable
  public static LookupElement createCompletionVariant(@NotNull SpringBeanPointer pointer, @NotNull String name) {
    if (!pointer.isValid()) return null;

    PsiElement element = pointer.getPsiElement();
    if (element instanceof PomTargetPsiElement) {
      final PomTarget target = ((PomTargetPsiElement)element).getTarget();
      if (target instanceof PsiTarget) {
        element = ((PsiTarget)target).getNavigationElement();
      }
    }
    if (element == null) {
      return null;
    }

    LookupElementBuilder lookupElement = LookupElementBuilder.create(element, name)
      .withIcon(SpringPresentationProvider.getSpringIcon(pointer));
    PsiClass beanClass = pointer.getBeanClass();
    if (beanClass != null) {
      lookupElement = lookupElement
        .withTypeText(beanClass.getName())
        .withStrikeoutness(beanClass.isDeprecated());
    }

    return lookupElement.withTailText(" (" + SpringPresentationProvider.getSpringBeanLocation(pointer) + ")", true);
  }

  public static boolean containsPatternReferences(@Nullable final String text) {
    return text != null && (StringUtil.containsChar(text, '*') || StringUtil.containsChar(text, '?'));
  }

  public static Collection<PsiPackage> getPackages(@Nullable final String text, @NotNull Project project) {
    return getPackages(text, ",; \n\t", project);
  }

  public static Collection<PsiPackage> getPackages(@Nullable final String text, @NotNull String delimiters, @NotNull Project project) {
    if (StringUtil.isEmptyOrSpaces(text)) {
      return Collections.emptySet();
    }
    final JavaPsiFacade psiFacade = JavaPsiFacade.getInstance(project);
    final List<PsiPackage> list = new SmartList<>();
    new DelimitedListProcessor(delimiters) {
      protected void processToken(final int start, final int end, final boolean delimitersOnly) {
        final String packageName = text.substring(start, end);
        ContainerUtil.addIfNotNull(list, psiFacade.findPackage(packageName.trim()));
      }
    }.processText(text);

    return list;
  }
}
