/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementResolveResult;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.ResolveResult;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PsiPackageReference;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.util.CommonProcessors;
import com.intellij.util.PatternUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.regex.Pattern;

public class SpringAntPatternPackageReferenceSet extends PackageReferenceSet {

  public SpringAntPatternPackageReferenceSet(@NotNull String packageName,
                                             @NotNull PsiElement element,
                                             int startInElement,
                                             @NotNull GlobalSearchScope scope) {
    super(packageName, element, startInElement, scope);
  }


  @NotNull
  @Override
  protected PsiPackageReference createReference(TextRange range, int index) {
    return new PsiPackageReference(this, range, index) {
      @NotNull
      @Override
      protected ResolveResult[] doMultiResolve() {
        Collection<PsiPackage> packages = new HashSet<>();
        for (PsiPackage parentPackage : getContext()) {
          packages.addAll(resolvePackages(parentPackage));
        }
        return PsiElementResolveResult.createResults(packages);
      }

      public Collection<PsiPackage> resolvePackages(@Nullable final PsiPackage context) {

        if (context == null) return Collections.emptySet();
        final String packageName = getValue();

        CommonProcessors.CollectProcessor<PsiPackage> processor = getProcessor(packageName);

        PsiPackageReference parentContextReference = myIndex > 0 ? getReferenceSet().getReference(myIndex - 1) : null;

        if (packageName.equals("*")) {
          for (PsiPackage aPackage : context.getSubPackages(getResolveScope())) {
            if (!processor.process(aPackage)) break;
          }
          return processor.getResults();
        }
        if (parentContextReference != null && parentContextReference.getValue().equals(("**"))) {
          return getSubPackages(context, processor);
        }

        if (packageName.equals("**")) {
          if (isLastReference()) {
            return getSubPackages(context, processor);
          }
          else {
            return Collections.singleton(context);
          }
        }
        else if ((packageName.contains("*") || packageName.contains("?"))) {
          for (final PsiPackage aPackage : context.getSubPackages(getResolveScope())) {
            processor.process(aPackage);
          }
          return processor.getResults();
        }

        return getReferenceSet().resolvePackageName(context, packageName);
      }

      public boolean isLastReference() {
        return this.equals(getReferenceSet().getLastReference());
      }

      @NotNull
      public CommonProcessors.CollectProcessor<PsiPackage> getProcessor(@NotNull String packageName) {
        final Pattern pattern = PatternUtil.fromMask(packageName);
        return new CommonProcessors.CollectProcessor<PsiPackage>(new LinkedHashSet<>()) {
          @Override
          protected boolean accept(PsiPackage aPackage) {
            String name = aPackage.getName();
            return name != null && pattern.matcher(name).matches();
          }
        };
      }

      @NotNull
      public Collection<PsiPackage> getSubPackages(@Nullable PsiPackage context, CommonProcessors.CollectProcessor<PsiPackage> processor) {
        processSubPackages(context, processor);
        return processor.getResults();
      }


      private void processSubPackages(final PsiPackage pkg,
                                      final CommonProcessors.CollectProcessor<PsiPackage> processor) {
        for (final PsiPackage aPackage : pkg.getSubPackages(getResolveScope())) {
          processor.process(aPackage);
          processSubPackages(aPackage, processor);
        }
      }
    };
  }
}