/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Comparing;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.qualifiers.SpringJamQualifier;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import gnu.trove.THashSet;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

public final class SpringQualifierComparator {

  public static boolean compareQualifiers(@Nullable SpringQualifier one, @Nullable final SpringQualifier two) {
    if (one == null || two == null) return false;
    if (!Comparing.equal(one.getQualifierType(), two.getQualifierType())) return false;
    if (!Comparing.equal(one.getQualifierValue(), two.getQualifierValue())) return false;
    final List<? extends QualifierAttribute> list1 = one.getQualifierAttributes();
    final int size1 = list1.size();
    final List<? extends QualifierAttribute> list2 = two.getQualifierAttributes();
    final int size2 = list2.size();
    if (size1 != size2) return false;
    if (size1 == 0) return true;
    final THashSet<QualifierAttribute> set = new THashSet<>(QualifierAttribute.HASHING_STRATEGY);
    set.addAll(list1);
    return set.containsAll(list2);
  }

  public static boolean compareInheritorQualifier(@Nullable SpringQualifier childrenQualifier,
                                                  @Nullable final SpringQualifier baseQualifier,
                                                  @Nullable Module module) {
    if (childrenQualifier instanceof SpringJamQualifier &&
        baseQualifier instanceof SpringJamQualifier) {
      if (Comparing.equal(childrenQualifier.getQualifierType(), baseQualifier.getQualifierType())) return false;

      if (module == null) return false;

      final PsiClass baseType = baseQualifier.getQualifierType();
      final PsiClass childrenType = childrenQualifier.getQualifierType();
      if (baseType != null && childrenType != null) {
        final String baseAnnoQualifiedName = baseType.getQualifiedName();
        if (baseAnnoQualifiedName != null) {
          final Collection<PsiClass> children =
            JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(baseAnnoQualifiedName);
          final PsiAnnotation definingMetaAnnotation =
            SpringAliasForUtils.findDefiningMetaAnnotation(childrenType, baseAnnoQualifiedName, children);
          if (definingMetaAnnotation != null) {
            return compareQualifiers(new SpringJamQualifier(definingMetaAnnotation, null), baseQualifier);
          }
        }
      }
    }

    return false;
  }
}
