/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts.propertySources;

import com.intellij.jam.JamConverter;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.*;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.psi.PsiAnchor;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.converters.PropertiesFileConverter;
import com.intellij.spring.model.jam.stereotype.SpringPropertySource;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Set;

/**
 * @since 2017.1
 */
public class CustomTestPropertySource implements SpringPropertySource {

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = JamService.ANNO_META_KEY.subKey("CustomTestPropertySource");
  public static final SemKey<CustomTestPropertySource> JAM_KEY =
    SpringPropertySource.PROPERTY_SOURCE_JAM_KEY.subKey("CustomTestPropertySource");
  public static final SemKey<JamMemberMeta<PsiClass, CustomTestPropertySource>> META_KEY = PROPERTY_SOURCE_META_KEY.subKey("CustomTestPropertySource");

  String[] LOCATIONS_FILES_ATTRS = {"locations", "value"};

  protected final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final AnnotationChildLink myAnnotationChildLink;
  private final PsiAnchor myPsiClassAnchor;

  private final NullableLazyValue<SpringTestPropertySource> myDefiningMetaAnnotation =
    new NullableLazyValue<SpringTestPropertySource>() {
      @Nullable
      @Override
      protected SpringTestPropertySource compute() {
        final PsiAnnotation definingMetaAnnotation = SpringAliasForUtils
          .findDefiningMetaAnnotation(getPsiElement(), myAnnotationChildLink.getAnnotationQualifiedName(),
                                      SpringTestPropertySource.ANNO_META.getAnnoName());
        if (definingMetaAnnotation != null) {
          final PsiClass annotationType = PsiTreeUtil.getParentOfType(definingMetaAnnotation, PsiClass.class, true);
          if (annotationType != null) {
            return SpringTestPropertySource.META.getJamElement(annotationType);
          }
        }
        return null;
      }
    };

  public CustomTestPropertySource(@NotNull String anno, @NotNull PsiClass psiClassAnchor) {
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiClassAnchor = PsiAnchor.create(psiClassAnchor);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(getPsiElement());
  }

  @Override
  public PsiClass getPsiElement() {
    return (PsiClass)myPsiClassAnchor.retrieve();
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Override
  public Set<PropertiesFile> getPropertiesFiles() {
    Set<PropertiesFile> files = ContainerUtil.newLinkedHashSet();

    JamConverter<Set<PropertiesFile>> referenceConverter = new PropertiesFileConverter();
    for (String attrName : LOCATIONS_FILES_ATTRS) {
      SpringAliasFor aliasFor = getAliasAttribute(attrName);
      if (aliasFor != null) {
        JamStringAttributeMeta.Collection<Set<PropertiesFile>> element = JamAttributeMeta.collectionString(aliasFor.getMethodName(), referenceConverter);
        for (JamStringAttributeElement<Set<PropertiesFile>> attributeElement : element.getJam(myPsiAnnotation)) {
          Set<PropertiesFile> values = attributeElement.getValue();
          if (values != null) files.addAll(values);
        }
        return files;
      }
    }

    final SpringTestPropertySource testPropertySource = getDefiningTestPropertySource();
    if (testPropertySource != null) {
      return testPropertySource.getPropertiesFiles();
    }
    return Collections.emptySet();
  }

  private SpringTestPropertySource getDefiningTestPropertySource() {
    return myDefiningMetaAnnotation.getValue();
  }

  private SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    return SpringAliasForUtils.findAliasFor(getPsiElement(),
                                            myAnnotationChildLink.getAnnotationQualifiedName(),
                                            SpringTestPropertySource.ANNO_META.getAnnoName(),
                                            attrName);
  }

  public boolean isPsiValid() {
    return getPsiElement().isValid();
  }
}