/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.vfs;

import com.intellij.database.model.DasObject;
import com.intellij.database.model.ObjectKind;
import com.intellij.database.util.DasUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.JBIterable;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author gregsh
 */
public final class ObjectPath {

  public final ObjectKind kind;
  public final List<String> path;

  public ObjectPath(@NotNull ObjectKind kind, @NotNull List<String> path) {
    this.kind = kind;
    this.path = ContainerUtil.immutableList(ArrayUtil.toStringArray(path));
  }

  @NotNull
  public String getDisplayName() {
    return StringUtil.join(JBIterable.from(path).filter(StringUtil::isNotEmpty), ".");
  }

  @NotNull
  public String getName() {
    return path.isEmpty() ? "" : path.get(path.size() - 1);
  }

  @Contract("!null->!null")
  public static ObjectPath of(@Nullable DasObject o) {
    if (o == null) return null;
    List<DasObject> parents = ContainerUtil.reverse(DasUtil.dasParents(o).filter(x -> x.getKind() != ObjectKind.ROOT).toList());
    return new ObjectPath(o.getKind(), JBIterable.from(parents).transform(DasUtil.TO_NAME).toList());
  }


  @Override
  public String toString() {
    return kind + ":" + path;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    ObjectPath path1 = (ObjectPath)o;

    if (!kind.equals(path1.kind)) return false;
    if (!path.equals(path1.path)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = kind.hashCode();
    result = 31 * result + path.hashCode();
    return result;
  }

}
