/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.codeHighlighting.HighlightDisplayLevel;
import com.intellij.codeInspection.LocalInspectionTool;
import com.intellij.codeInspection.ProblemDescriptor;
import com.intellij.openapi.compiler.CompileContext;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiFile;
import org.jetbrains.annotations.TestOnly;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

/**
 * Registry of all configuration file inspections provided by Spring-* plugins.
 *
 * @author Yann C&eacute;bron
 */
public abstract class SpringInspectionsRegistry {

  /**
   * Provide plugin-specific configuration file inspections.
   */
  public interface Contributor {

    ExtensionPointName<SpringInspectionsRegistry.Contributor> EP_NAME =
      ExtensionPointName.create("com.intellij.spring.inspectionsRegistryContributor");

    Class<? extends LocalInspectionTool>[] getInspectionClasses();
  }

  /**
   * Allows adding additional configuration files as well as additional highlighting to Spring validation context.
   *
   * @since 14.1
   */
  public abstract static class AdditionalFilesContributor {

    public static final ExtensionPointName<SpringInspectionsRegistry.AdditionalFilesContributor> EP_NAME =
      ExtensionPointName.create("com.intellij.spring.inspectionsRegistryAdditionalFilesContributor");

    public abstract Collection<VirtualFile> getAdditionalFilesToProcess(final Project project, final CompileContext context);

    /**
     * Add additional highlighting (e.g. provided by Annotators) for the given file.
     * <p/>
     * NOTE: all files from current validation context will be passed in, not only additional ones provided by this EP.
     *
     * @param psiFile File to check.
     * @return Additional highlighting info.
     */
    public Map<ProblemDescriptor, HighlightDisplayLevel> checkAdditionally(PsiFile psiFile) {
      return Collections.emptyMap();
    }
  }

  public static SpringInspectionsRegistry getInstance() {
    return ServiceManager.getService(SpringInspectionsRegistry.class);
  }

  /**
   * Returns all registered inspections. Used by Compiler|Validation "Spring Model".
   *
   * @return Inspections.
   */
  public abstract Class<? extends LocalInspectionTool>[] getSpringInspectionClasses();

  @TestOnly
  public abstract Class<? extends LocalInspectionTool>[] getTestSpringInspectionClasses();

  @TestOnly
  public abstract Class<? extends LocalInspectionTool> getTestSpringModelInspectionClass();
}
