/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.pom.references.PomService;
import com.intellij.psi.*;
import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * Used for beans created implicitly by other bean
 *
 * @since 14
 */
public class SpringImplicitBeanWithDefinition extends CommonModelElement.PsiBase
  implements CommonSpringBean {

  @NotNull
  private final String myName;
  @NotNull
  private final PsiClass myClass;
  @NotNull
  private final CommonSpringBean myDefiningBean;
  @NotNull
  private final PsiTarget myDefinitionTarget;

  /**
   * @param beanName
   * @param beanClass
   * @param definingBean     bean that implicitly creates this bean
   * @param definitionTarget target to psi element that is definition of this bean
   */
  public SpringImplicitBeanWithDefinition(@NotNull String beanName,
                                          @NotNull PsiClass beanClass,
                                          @NotNull CommonSpringBean definingBean,
                                          @NotNull PsiTarget definitionTarget) {
    myName = beanName;
    myClass = beanClass;
    myDefiningBean = definingBean;
    myDefinitionTarget = definitionTarget;
  }

  @Nullable
  @Override
  public String getBeanName() {
    return myName;
  }

  @NotNull
  @Override
  public String[] getAliases() {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  @Nullable
  @Override
  public PsiType getBeanType(boolean considerFactories) {
    return getBeanType();
  }

  @Nullable
  @Override
  public PsiType getBeanType() {
    return JavaPsiFacade.getElementFactory(myClass.getProject()).createType(myClass);
  }

  @NotNull
  @Override
  public Collection<SpringQualifier> getSpringQualifiers() {
    return Collections.singleton(DefaultSpringBeanQualifier.create(this));
  }

  @NotNull
  @Override
  public SpringProfile getProfile() {
    return myDefiningBean.getProfile();
  }

  @Override
  public boolean isPrimary() {
    return false;
  }

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return myDefinitionTarget.getNavigationElement();
  }

  @Override
  public PsiElement getIdentifyingPsiElement() {
    return PomService.convertToPsi(myDefiningBean.getPsiManager().getProject(), myDefinitionTarget);
  }
}
