/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.spring.model.utils.resources;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.psi.ElementManipulators;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.util.Function;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public class SpringResourcesBuilder {

  @NotNull private final PsiElement myElement;
  @NotNull private final String myText;
  private boolean myFromRoot = false;
  private boolean myFromCurrent = false;
  private int myOffset = 0;
  private boolean mySoft = true;
  @NotNull private Condition<PsiFileSystemItem> myFilter = Conditions.alwaysTrue();
  private boolean myEndingSlashNotAllowed = true;
  @Nullable private Function<PsiFile, Collection<PsiFileSystemItem>> myCustomDefaultPathEvaluator = null;
  @NotNull private Module[] myModules = Module.EMPTY_ARRAY;
  @Nullable private String myTemplateName;

  private SpringResourcesBuilder(final @NotNull PsiElement element,
                                 final @NotNull String text) {
    this(element, text, ElementManipulators.getOffsetInElement(element));
  }

  private SpringResourcesBuilder(final @NotNull PsiElement element,
                                 final @NotNull String text,
                                 int offset) {
    myElement = element;
    myText = text;
    myOffset = offset;
  }

  public static SpringResourcesBuilder create(final @NotNull PsiElement element,
                                              final @NotNull String s) {
    return new SpringResourcesBuilder(element, s);
  }

  public static SpringResourcesBuilder create(final @NotNull PsiElement element,
                                              final @NotNull String s,
                                              int offset) {
    return new SpringResourcesBuilder(element, s, offset);
  }

  public SpringResourcesBuilder fromRoot(boolean fromRoot) {
    myFromRoot = fromRoot;
    return this;
  }

  public SpringResourcesBuilder fromCurrent(boolean fromCurrent) {
    myFromCurrent = fromCurrent;
    return this;
  }

  public SpringResourcesBuilder offset(int offset) {
    myOffset = offset;
    return this;
  }

  public SpringResourcesBuilder soft(boolean soft) {
    mySoft = soft;
    return this;
  }

  public SpringResourcesBuilder endingSlashNotAllowed(boolean endingSlashNotAllowed) {
    myEndingSlashNotAllowed = endingSlashNotAllowed;
    return this;
  }

  public SpringResourcesBuilder filter(@NotNull Condition<PsiFileSystemItem> filter) {
    myFilter = filter;
    return this;
  }

  public SpringResourcesBuilder customDefaultPathEvaluator(@Nullable Function<PsiFile, Collection<PsiFileSystemItem>> customDefaultPathEvaluator) {
    myCustomDefaultPathEvaluator = customDefaultPathEvaluator;
    return this;
  }

  public SpringResourcesBuilder modules(@NotNull Module... modules) {
    myModules = modules;
    return this;
  }

  /**
   * Sets template name to use when creating not existing file.
   *
   * @param templateName File template name.
   * @return This.
   * @see com.intellij.ide.fileTemplates.FileTemplateManager
   * @since 2017.2
   */
  public SpringResourcesBuilder newFileTemplateName(String templateName) {
    myTemplateName = templateName;
    return this;
  }

  @NotNull
  PsiElement getElement() {
    return myElement;
  }

  @NotNull
  String getText() {
    return myText;
  }

  boolean isFromRoot() {
    return myFromRoot;
  }

  boolean isFromCurrent() {
    return myFromCurrent;
  }

  int getOffset() {
    return myOffset;
  }

  boolean isSoft() {
    return mySoft;
  }

  @NotNull
  Condition<PsiFileSystemItem> getFilter() {
    return myFilter;
  }

  boolean isEndingSlashNotAllowed() {
    return myEndingSlashNotAllowed;
  }

  @Nullable
  Function<PsiFile, Collection<PsiFileSystemItem>> getCustomDefaultPathEvaluator() {
    return myCustomDefaultPathEvaluator;
  }

  @NotNull
  Module[] getModules() {
    return myModules;
  }

  @Nullable
  String getTemplateName() {
    return myTemplateName;
  }
}
