/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.references.property;

import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.LocalQuickFixProvider;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.psi.impl.beanProperties.BeanProperty;
import com.intellij.psi.impl.beanProperties.CreateBeanPropertyFixes;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.util.PropertyUtil;
import com.intellij.psi.util.PropertyUtilBase;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author Yann C&eacute;bron
 * @since 2017.1
 */
class BindingPropertyReference extends PsiReferenceBase<PsiElement> implements LocalQuickFixProvider, EmptyResolveMessageProvider {

  private final BindingPropertyReferenceSet mySet;
  private final int myIndex;
  private final boolean myHasIndexAccess;

  BindingPropertyReference(BindingPropertyReferenceSet set,
                           TextRange range,
                           int index) {
    super(set.getElement(), range);

    mySet = set;
    myIndex = index;

    final String value = getValue();
    final int indexStartIdx = value.indexOf("[");
    myHasIndexAccess = indexStartIdx != -1 && value.endsWith("]");
    if (myHasIndexAccess) {
      setRangeInElement(TextRange.from(range.getStartOffset(), indexStartIdx));
    }
  }

  @Override
  public boolean isSoft() {
    return mySet.isSoft();
  }

  @Nullable
  private PsiClass getPsiClass() {
    if (isFirst()) {
      return mySet.getModelClass();
    }

    final BindingPropertyReference reference = mySet.getReference(myIndex - 1);
    final PsiMethod psiMethod = (PsiMethod)reference.resolve();
    if (psiMethod == null) {
      return null;
    }

    PsiType type = psiMethod.getReturnType();
    if (type == null) {
      return null;
    }

    final boolean usingIndexAccess = reference.hasIndexAccess();
    if (usingIndexAccess) {
      type = type.getDeepComponentType();
    }
    if (type instanceof PsiClassType) {
      final PsiClassType psiClassType = (PsiClassType)type;
      final PsiClass psiClass = psiClassType.resolve();

      if (usingIndexAccess &&
          psiClassType.getParameterCount() == 1 &&
          InheritanceUtil.isInheritor(psiClass, CommonClassNames.JAVA_UTIL_LIST) ||
          InheritanceUtil.isInheritor(psiClass, CommonClassNames.JAVA_UTIL_SET)) {
        final PsiType collectionElementType = psiClassType.getParameters()[0];
        return collectionElementType instanceof PsiClassType ? ((PsiClassType)collectionElementType).resolve() : null;
      }

      return psiClass;
    }
    return null;
  }

  private boolean isFirst() {
    return myIndex == 0;
  }

  private boolean hasIndexAccess() {
    return myHasIndexAccess;
  }

  @Override
  public PsiElement resolve() {
    final PsiClass psiClass = getPsiClass();
    final String propertyName = getValue();

    return PropertyUtilBase.findPropertyGetter(psiClass, propertyName, false, true);
  }

  @NotNull
  @Override
  public Object[] getVariants() {
    return ContainerUtil.map2Array(resolveProperties(), PsiNamedElement.class, BeanProperty::getPsiElement);
  }

  private List<BeanProperty> resolveProperties() {
    final PsiClass psiClass = getPsiClass();
    if (psiClass == null) {
      return Collections.emptyList();
    }

    final Map<String, PsiMethod> properties = PropertyUtilBase.getAllProperties(psiClass, false, true, true);
    return ContainerUtil.mapNotNull(properties.values(),
                                    (NullableFunction<PsiMethod, BeanProperty>)BeanProperty::createBeanProperty);
  }

  public PsiElement handleElementRename(final String newElementName) throws IncorrectOperationException {
    final String name = PropertyUtilBase.getPropertyName(newElementName);
    return super.handleElementRename(name == null ? newElementName : name);
  }

  public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
    if (element instanceof PsiMethod) {
      final String propertyName = PropertyUtilBase.getPropertyName((PsiMember)element);
      if (propertyName != null) {
        return super.handleElementRename(propertyName);
      }
    }
    return getElement();
  }

  @Override
  public LocalQuickFix[] getQuickFixes() {
    if (resolve() != null) {
      return LocalQuickFix.EMPTY_ARRAY;
    }
    final PsiClass psiClass = getPsiClass();
    if (psiClass == null) {
      return LocalQuickFix.EMPTY_ARRAY;
    }
    return CreateBeanPropertyFixes.createFixes(getValue(), psiClass, null, false);
  }

  @NotNull
  @Override
  public String getUnresolvedMessagePattern() {
    return "Cannot resolve property '" + getValue() + "'";
  }
}