/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.Processor;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public interface CommonSpringModel {

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   * @since 14
   */
  boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params, @NotNull Processor<SpringBeanPointer> processor);

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   * @since 14
   */
  boolean processByName(@NotNull SpringModelSearchParameters.BeanName params, @NotNull Processor<SpringBeanPointer> processor);


  /**
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @since 2017.3
   */
  default boolean processAllBeans(@NotNull Processor<SpringBeanPointer> processor) {return true;};

    /**
     * @return All beans configured in XML files.
     */
  @NotNull
  Collection<SpringBeanPointer> getAllDomBeans();

  /**
   * NOTE: Expensive operation. Consider using {@code process...()} methods instead.
   *
   * @return All beans.
   */
  @NotNull
  Collection<SpringBeanPointer> getAllCommonBeans();

  @NotNull
  Set<String> getAllBeanNames(@NotNull final String beanName);

  Collection<PsiElementPointer> getDuplicatedNames(@NotNull String beanName);

  @NotNull
  List<SpringBeanPointer> getPlaceholderConfigurers();

  @NotNull
  List<? extends SpringBeansPackagesScan> getComponentScans();

  @NotNull
  List<SpringBeanPointer> getAnnotationConfigApplicationContexts();

  Collection<XmlTag> getCustomBeanCandidates(String id);

  /**
   * Returns the associated module.
   *
   * @return Module containing this model or {@code null} if model is not bound to a specific module (e.g. Project-global).
   */
  @Nullable
  Module getModule();

  @Nullable
  Set<String> getActiveProfiles();

  @NotNull
  Set<String> getAllProfiles();

  /**
   * @return All configuration files.
   * @see #hasConfigFile(PsiFile)
   */
  @NotNull
  Set<PsiFile> getConfigFiles();

  /**
   * Returns whether this model uses the given config file.
   *
   * @param configFile Config file.
   * @return {@code true} if file belongs to model.
   * @since 14
   */
  boolean hasConfigFile(@NotNull PsiFile configFile);

  @NotNull
  List<SpringBeanPointer> findQualifiedBeans(@NotNull SpringQualifier qualifier);

  /**
   * Returns descendant beans having given context as "parent".
   *
   * @param context Bean to get descendants for.
   * @return Empty list by default.
   */
  @NotNull
  default List<SpringBeanPointer> getDescendants(final @NotNull SpringBeanPointer context) {
    return Collections.emptyList();
  }
}
