/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.SpringJamModel;
import com.intellij.spring.model.jam.javaConfig.SpringJavaBean;
import com.intellij.spring.model.jam.stereotype.SpringConfiguration;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.SpringJamUtils;
import com.intellij.spring.model.jam.utils.filters.SpringContextFilter;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.NotNullFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class ComponentScanPackagesModel extends AbstractSimpleSpringModel {

  private final NotNullLazyValue<Set<PsiPackage>> myPackages;

  @NotNull
  private final Module myModule;

  public ComponentScanPackagesModel(@NotNull NotNullLazyValue<Set<PsiPackage>> packages, @NotNull Module module) {
    myPackages = packages;
    myModule = module;
  }

  @Override
  protected final Collection<SpringBeanPointer> calculateLocalBeans() {
    Collection<SpringBeanPointer> pointers = calculateScannedBeans();

    Set<SpringJavaBean> javaBeans = ContainerUtil.newLinkedHashSet();
    for (SpringBeanPointer pointer : pointers) {
      CommonSpringBean springBean = pointer.getSpringBean();
      if (springBean instanceof SpringConfiguration) continue;
      if (springBean instanceof SpringStereotypeElement) {
        javaBeans.addAll(((SpringStereotypeElement)springBean).getBeans());
      }
    }

    Set<SpringBeanPointer> beans = ContainerUtil.newLinkedHashSet();
    beans.addAll(pointers);
    beans.addAll(BeanService.getInstance().mapSpringBeans(javaBeans));

    return beans;
  }

  @NotNull
  @Override
  public Collection<SpringBeanPointer> calculateDomBeans() {
    return Collections.emptySet();
  }

  protected Collection<SpringBeanPointer> calculateScannedBeans() {
    return getScannedComponents(myPackages.getValue(), getModule(), getActiveProfiles());
  }

  public static Collection<SpringBeanPointer> getScannedComponents(@NotNull Set<PsiPackage> packages,
                                                                   @NotNull Module module,
                                                                   @Nullable Set<String> profiles) {
    return getScannedComponents(packages, module, profiles, true,
                                Collections.emptySet(),
                                Collections.emptySet());
  }

  public static Collection<SpringConfiguration> getScannedConfigurations(SpringBeansPackagesScan scan,
                                                                         @NotNull final Module module,
                                                                         @Nullable Set<String> profiles) {

    List<CommonSpringBean> components = getScannedComponents(dom -> SpringJamModel.getModel(module).getConfigurations(dom),
                                                             scan.getPsiPackages(), module, profiles, scan.useDefaultFilters(),
                                                             scan.getExcludeContextFilters(),
                                                             scan.getIncludeContextFilters());

    Set<SpringConfiguration> configurations = new LinkedHashSet<>();
    for (CommonSpringBean component : components) {
      if (component instanceof SpringConfiguration) {
        configurations.add((SpringConfiguration)component);
      }
    }
    return configurations;
  }

  public static Collection<SpringBeanPointer> getScannedComponents(@NotNull Set<PsiPackage> packages,
                                                                   @NotNull final Module module,
                                                                   @Nullable Set<String> profiles,
                                                                   boolean useDefaultFilters,
                                                                   @NotNull Set<SpringContextFilter.Exclude> excludeContextFilters,
                                                                   @NotNull Set<SpringContextFilter.Include> includeContextFilters) {
    Collection<CommonSpringBean> components =
      getScannedComponents(scope -> SpringJamModel.getModel(module).getStereotypeComponents(scope),
                           packages, module, profiles,
                           useDefaultFilters, excludeContextFilters, includeContextFilters);

    return BeanService.getInstance().mapSpringBeans(components);
  }

  public static <T extends SpringStereotypeElement> List<CommonSpringBean> getScannedComponents(@NotNull NotNullFunction<GlobalSearchScope, List<T>> components,
                                                                                                @NotNull Set<PsiPackage> packages,
                                                                                                @NotNull Module module,
                                                                                                @Nullable Set<String> profiles,
                                                                                                boolean useDefaultFilters,
                                                                                                @NotNull Set<SpringContextFilter.Exclude> excludeContextFilters,
                                                                                                @NotNull Set<SpringContextFilter.Include> includeContextFilters) {
    if (module.isDisposed() || packages.isEmpty()) return ContainerUtil.newSmartList();

    GlobalSearchScope allPackagesUnionScope = null;
    for (PsiPackage psiPackage : packages) {

      if (psiPackage.getQualifiedName().isEmpty() &&
          ApplicationManager.getApplication().isUnitTestMode()) {
        throw new IllegalArgumentException("Do not use component-scan with <default> package in tests");
      }

      final GlobalSearchScope packageScope = PackageScope.packageScope(psiPackage, true);
      allPackagesUnionScope = allPackagesUnionScope == null ? packageScope : allPackagesUnionScope.uniteWith(packageScope);
    }

    final GlobalSearchScope moduleScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
    final GlobalSearchScope effectiveSearchScope = moduleScope.intersectWith(allPackagesUnionScope);

    final List<T> allPointers = components.fun(effectiveSearchScope);

    Set<CommonSpringBean> filteredBeans =
      SpringJamUtils.getInstance().filterComponentScannedStereotypes(module, allPointers, packages, useDefaultFilters,
                                                                     excludeContextFilters, includeContextFilters);

    return SpringProfileUtils.filterBeansInActiveProfiles(filteredBeans, profiles);
  }

  @NotNull
  @Override
  public Module getModule() {
    return myModule;
  }

  @NotNull
  @Override
  public Set<String> getAllProfiles() {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public Set<PsiFile> getConfigFiles() {
    return Collections.emptySet();
  }

  @Override
  public boolean hasConfigFile(@NotNull PsiFile configFile) {
    return false;
  }
}
