/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.SystemProperties;
import com.intellij.util.containers.SLRUMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

/**
 * @since 2016.3
 */
public abstract class SpringCachingProcessor<InParams extends SpringModelSearchParameters> {

  private static final int DEFAULT_CACHE_SIZE = 1000;
  private static final String CACHE_SIZE_PROPERTY_NAME = "idea.spring.model.processing.cache.size";
  private static final int CONFIGURED_CACHE_SIZE = SystemProperties.getIntProperty(CACHE_SIZE_PROPERTY_NAME, DEFAULT_CACHE_SIZE);

  private final SpringSLRUCache<InParams, Collection<SpringBeanPointer>> myFindAllCache;
  private final SpringSLRUCache<InParams, NullableLazyValue<SpringBeanPointer>> myFindFirstCache;

  protected SpringCachingProcessor() {
    myFindAllCache = new SpringSLRUCache<InParams, Collection<SpringBeanPointer>>(CONFIGURED_CACHE_SIZE) {
      @NotNull
      @Override
      protected Collection<SpringBeanPointer> createValue(InParams key) {
        return findPointers(key);
      }
    };

    myFindFirstCache = new SpringSLRUCache<InParams, NullableLazyValue<SpringBeanPointer>>(CONFIGURED_CACHE_SIZE) {
      @NotNull
      @Override
      protected NullableLazyValue<SpringBeanPointer> createValue(InParams key) {
        return new NullableLazyValue<SpringBeanPointer>() {
          @Nullable
          @Override
          protected SpringBeanPointer compute() {
            return findFirstPointer(key);
          }
        };
      }
    };
  }

  @NotNull
  protected abstract Collection<SpringBeanPointer> findPointers(@NotNull InParams parameters);

  @Nullable
  protected abstract SpringBeanPointer findFirstPointer(@NotNull InParams parameters);

  public boolean process(InParams params,
                         final Processor<SpringBeanPointer> processor,
                         final Set<String> activeProfiles) {
    if (processor instanceof CommonProcessors.FindFirstProcessor && myFindAllCache.getCachedValue(params) == null) {
      final SpringBeanPointer first = myFindFirstCache.get(params).getValue();
      return processBeansInActiveProfile(processor, first, activeProfiles);
    }

    for (SpringBeanPointer pointer : myFindAllCache.get(params)) {
      if (!processBeansInActiveProfile(processor, pointer, activeProfiles)) return false;
    }
    return true;
  }

  private static boolean processBeansInActiveProfile(Processor<SpringBeanPointer> processor,
                                                     @Nullable SpringBeanPointer pointer,
                                                     Set<String> activeProfiles) {
    if (pointer == null) return true;
    if (activeProfiles == null || activeProfiles.isEmpty()) return processor.process(pointer);

    if (!SpringProfileUtils.isProfileAccepted(pointer.getSpringBean().getProfile(), activeProfiles)) return true;

    return processor.process(pointer);
  }


  protected abstract static class SpringSLRUCache<K, V> extends SLRUMap<K, V> {
    private final Object myLock = new Object();

    protected SpringSLRUCache(final int queueSize) {
      super(queueSize, queueSize);
    }

    @NotNull
    protected abstract V createValue(K key);

    @Override
    @NotNull
    public V get(K key) {
      V value = getCachedValue(key);
      if (value != null) return value;

      value = createValue(key);
      synchronized (myLock) {
        put(key, value);
      }
      return value;
    }

    @Nullable
    protected V getCachedValue(K key) {
      V value;
      synchronized (myLock) {
        value = super.get(key);
      }
      return value;
    }
  }
}
