/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet.beans;

import com.intellij.openapi.util.Key;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class CustomSetting {

  private String myName;
  private String myDescription;

  public String getName() {
    return myName;
  }

  public String getDescription() {
    return myDescription;
  }

  public abstract void setStringValue(@NotNull String value);

  @Nullable
  public abstract String getStringValue();

  public abstract boolean isModified();

  public abstract void apply();

  public abstract void reset();

  protected CustomSetting(String name, String description) {
    myName = name;
    myDescription = description;
  }

  protected CustomSetting(Key<? extends CustomSetting> key, String description) {
    myName = key.toString();
    myDescription = description;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    CustomSetting setting = (CustomSetting)o;

    if (!myName.equals(setting.myName)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return myName.hashCode();
  }

  /**
   * @since 2017.2
   */
  public static class STRING extends CustomSetting {

    private String myValue;
    private String myDefaultValue;
    private boolean myModified;

    public STRING(Key<CustomSetting.STRING> key, String description, String defaultValue) {
      super(key, description);
      myDefaultValue = defaultValue;
    }

    public void clear() {
      myValue = null;
    }

    @NotNull
    public String getDefaultValue() {
      return myDefaultValue;
    }

    @Override
    public void setStringValue(@NotNull String value) {
      if (!value.equals(myValue)) {
        myValue = value;
        myModified = true;
      }
    }

    @Nullable
    @Override
    public String getStringValue() {
      return myValue;
    }

    @Override
    public boolean isModified() {
      return myModified;
    }

    @Override
    public void apply() {
      myModified = false;
    }

    @Override
    public void reset() {
      if (myModified) myValue = null;
      myModified = false;
    }
  }

  public static class BOOLEAN extends CustomSetting {
    private Boolean myValue = null;
    private boolean myDefaultValue;
    private boolean myModified;

    public BOOLEAN(Key<CustomSetting.BOOLEAN> key, String description, boolean defaultValue) {
      super(key, description);
      myDefaultValue = defaultValue;
    }

    public void setBooleanValue(boolean value) {
      boolean currentValue = getBooleanValue();
      if (value != currentValue) {
        myModified = true;
        myValue = value;
      }
    }

    public boolean getBooleanValue() {
      return (myModified || myValue == null) ? myDefaultValue : myValue.booleanValue();
    }

    @Nullable
    public Boolean getValue() {
      return myValue;
    }

    public boolean getDefaultValue() {
      return myDefaultValue;
    }

    @Override
    public void setStringValue(@NotNull String value) {
      setBooleanValue(Boolean.valueOf(value));
    }

    @Nullable
    @Override
    public String getStringValue() {
      return myValue == null ? null : Boolean.toString(myValue);
    }

    @Override
    public void apply() {
      myModified = false;
    }

    @Override
    public void reset() {
      if (myModified) myValue = null;
      myModified = false;
    }

    public boolean isModified() {
      return myModified;
    }
  }
}
