/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.gutter;

import com.intellij.ide.util.PsiElementListCellRenderer;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.PsiElement;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomManager;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

/**
 * @author Dmitry Avdeev
 */
public class DomElementListCellRenderer extends PsiElementListCellRenderer<PsiElement> {

  protected final String myUnknown;

  public DomElementListCellRenderer(@Nls String unknownElementText) {
    myUnknown = unknownElementText;
  }

  public String getElementText(final PsiElement element) {
    String elementName = null;

    final DomElement domElement = getDomElement(element);
    if (domElement != null) {
      elementName = domElement.getPresentation().getElementName();
    }
    else if (element instanceof XmlTag) {
      return ((XmlTag)element).getName();
    }

    return elementName == null ? myUnknown : elementName;
  }

  @Nullable
  public static DomElement getDomElement(@Nullable PsiElement element) {
    if (element instanceof PomTargetPsiElement) {
      return getDomElement(element.getNavigationElement());
    }
    else if (element instanceof XmlTag) {
      return DomManager.getDomManager(element.getProject()).getDomElement((XmlTag)element);
    }
    return null;
  }

  protected String getContainerText(final PsiElement element, final String name) {
    return getContainerText(element);
  }

  public static String getContainerText(final PsiElement element) {
    return " (" + element.getContainingFile().getName() + ")";
  }

  protected int getIconFlags() {
    return 0;
  }

  protected Icon getIcon(final PsiElement element) {
    final DomElement domElement = getDomElement(element);

    if (domElement != null) {
      final Icon presentationIcon = domElement.getPresentation().getIcon();
      if (presentationIcon != null) {
        return presentationIcon;
      }
    }

    return super.getIcon(element);
  }
}