/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Pair;
import com.intellij.patterns.ElementPattern;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.semantic.SemKey;
import com.intellij.semantic.SemRegistrar;
import com.intellij.semantic.SemService;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Consumer;
import com.intellij.util.Function;
import com.intellij.util.NotNullFunction;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.MultiMap;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public final class SpringSemContributorUtil {

  private static final MultiMap<String, JamCustomImplementationBean> ourJamCustomImplementationBeans;

  private SpringSemContributorUtil() {
  }

  static {
    MultiMap<String, JamCustomImplementationBean> beans = MultiMap.createSmart();
    for (JamCustomImplementationBean bean : JamCustomImplementationBean.EP_NAME.getExtensions()) {
      beans.putValue(bean.baseAnnotationFqn, bean);
    }
    ourJamCustomImplementationBeans = beans;
  }

  public static <T extends JamElement, Psi extends PsiMember> void registerMetaComponents(@NotNull final SemService semService,
                                                                                          @NotNull SemRegistrar registrar,
                                                                                          @NotNull ElementPattern<? extends Psi> place,
                                                                                          @NotNull final SemKey<JamMemberMeta<Psi, T>> metaKey,
                                                                                          @NotNull final SemKey<T> semKey,
                                                                                          @NotNull final NullableFunction<Psi, JamMemberMeta<Psi, T>> metaFunction) {
    registrar.registerSemElementProvider(metaKey, place, metaFunction);

    registrar.registerSemElementProvider(semKey, place,
                                         (NullableFunction<Psi, T>)member -> {
                                           final JamMemberMeta<Psi, T> memberMeta = semService.getSemElement(metaKey, member);
                                           return memberMeta != null ? memberMeta.createJamElement(PsiElementRef.real(member)) : null;
                                         }
    );
  }

  public static <T extends JamElement, Psi extends PsiMember> void registerRepeatableMetaComponents(@NotNull final SemService semService,
                                                                                          @NotNull SemRegistrar registrar,
                                                                                          @NotNull ElementPattern<? extends Psi> place,
                                                                                          @NotNull final SemKey<JamMemberMeta<Psi, T>> metaKey,
                                                                                          @NotNull final SemKey<T> semKey,
                                                                                          @NotNull final NullableFunction<Psi, Collection<JamMemberMeta<Psi, T>>> metaFunction) {
    registrar.registerRepeatableSemElementProvider(metaKey, place, metaFunction);

    registrar.registerRepeatableSemElementProvider(semKey, place,
                                         (NullableFunction<Psi, Collection<T>>)member -> {
                                           final List<JamMemberMeta<Psi, T>> memberMetas = semService.getSemElements(metaKey, member);
                                           Collection<T> metas = ContainerUtil.newHashSet();
                                           for (JamMemberMeta<Psi, T> memberMeta : memberMetas) {
                                             ContainerUtil.addIfNotNull(metas, memberMeta.createJamElement(PsiElementRef.real(member)));
                                           }
                                           return metas.isEmpty() ? null : metas;
                                         }
    );
  }

  /**
   * @since 15
   */
  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, JamMemberMeta<Psi, T>> createFunction(@NotNull final SemKey<T> semKey,
                                                                                                                          @NotNull final Class<T> jamClass,
                                                                                                                          @NotNull final Function<Module, Collection<String>> annotationsGetter,
                                                                                                                          @NotNull final Function<Pair<String, Psi>, T> producer,
                                                                                                                          @Nullable final Consumer<JamMemberMeta<Psi, T>> metaConsumer) {
    return createFunction(semKey, jamClass, annotationsGetter, producer, metaConsumer, null);
  }

  /**
   * @see SpringAliasForUtils#getAnnotationMetaProducer(SemKey, JamMemberMeta[])
   * @since 2017.1
   */
  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, JamMemberMeta<Psi, T>> createFunction(@NotNull final SemKey<T> semKey,
                                                                                                                          @NotNull final Class<T> jamClass,
                                                                                                                          @NotNull final Function<Module, Collection<String>> annotationsGetter,
                                                                                                                          @NotNull final Function<Pair<String, Psi>, T> producer,
                                                                                                                          @Nullable final Consumer<JamMemberMeta<Psi, T>> metaConsumer,
                                                                                                                          @Nullable final NotNullFunction<Pair<String, Project>, JamAnnotationMeta> annotationMeta) {
    return psiMember -> {
      if (DumbService.isDumb(psiMember.getProject())) return null;
      if (psiMember instanceof PsiClass && ((PsiClass)psiMember).isAnnotationType()) return null;
      final Module module = ModuleUtilCore.findModuleForPsiElement(psiMember);
      for (final String anno : annotationsGetter.fun(module)) {
        if (AnnotationUtil.isAnnotated(psiMember, anno, true)) {
           return getMeta(semKey, jamClass, producer, metaConsumer, annotationMeta, psiMember, anno);
        }
      }
      return null;
    };
  }

  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, Collection<JamMemberMeta<Psi, T>>> createRepeatableFunction(@NotNull final SemKey<T> semKey,
                                                                                                                          @NotNull final Class<T> jamClass,
                                                                                                                          @NotNull final Function<Module, Collection<String>> annotationsGetter,
                                                                                                                          @NotNull final Function<Pair<String, Psi>, T> producer,
                                                                                                                          @Nullable final Consumer<JamMemberMeta<Psi, T>> metaConsumer,
                                                                                                                          @Nullable final NotNullFunction<Pair<String, Project>, JamAnnotationMeta> annotationMeta) {
    return psiMember -> {
      Collection<JamMemberMeta<Psi, T>> metas = ContainerUtil.newHashSet();
      if (DumbService.isDumb(psiMember.getProject())) return null;
      if (psiMember instanceof PsiClass && ((PsiClass)psiMember).isAnnotationType()) return null;
      final Module module = ModuleUtilCore.findModuleForPsiElement(psiMember);
      for (final String anno : annotationsGetter.fun(module)) {
        if (AnnotationUtil.isAnnotated(psiMember, anno, true)) {
          final JamMemberMeta<Psi, T> meta = getMeta(semKey, jamClass, producer, metaConsumer, annotationMeta, psiMember, anno);

         metas.add(meta);
        }
      }
      return metas.isEmpty() ? null : metas;
    };
  }

  @NotNull
  private static <T extends JamElement, Psi extends PsiMember> JamMemberMeta<Psi, T> getMeta(@NotNull SemKey<T> semKey,
                                                                                             @NotNull Class<T> jamClass,
                                                                                             @NotNull Function<Pair<String, Psi>, T> producer,
                                                                                             @Nullable Consumer<JamMemberMeta<Psi, T>> metaConsumer,
                                                                                             @Nullable NotNullFunction<Pair<String, Project>, JamAnnotationMeta> annotationMeta,
                                                                                             Psi psiMember, String anno) {
    final JamMemberMeta<Psi, T> meta = new JamMemberMeta<Psi, T>(null, jamClass, semKey) {
      @Override
      public T createJamElement(PsiElementRef<Psi> psiMemberPsiRef) {
        return producer.fun(Pair.create(anno, psiMemberPsiRef.getPsiElement()));
      }
    };
    if (metaConsumer != null) {
      metaConsumer.consume(meta);
    }
    if (annotationMeta != null) registerCustomAnnotationMeta(anno, meta, annotationMeta, psiMember.getProject());
    return meta;
  }

  private static <T extends JamElement, Psi extends PsiMember> void registerCustomAnnotationMeta(@NotNull String anno,
                                                                                                 @NotNull JamMemberMeta<Psi, T> meta,
                                                                                                 @NotNull NotNullFunction<Pair<String, Project>, JamAnnotationMeta> metaNotNullFunction,
                                                                                                 @NotNull Project project) {
    List<JamAnnotationMeta> annotations = meta.getAnnotations();
    for (JamAnnotationMeta annotationMeta : annotations) {
      if (anno.equals(annotationMeta.getAnnoName())) return;
    }
    meta.addAnnotation(metaNotNullFunction.fun(Pair.create(anno, project)));
  }

  /**
   * @param <T>
   * @return Consumer.
   * @see SpringStereotypeElement#addPomTargetProducer(JamMemberMeta)
   * @since 15
   */
  public static <T extends SpringStereotypeElement, Psi extends PsiMember> Consumer<JamMemberMeta<Psi, T>> createStereotypeConsumer() {
    return SpringStereotypeElement::addPomTargetProducer;
  }

  /**
   * @param anno Annotation FQN.
   * @return Custom annotation types.
   * @since 15
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno) {
    return getCustomMetaAnnotations(anno, false);
  }

  /**
   * Returns all custom meta annotations in defined scope.
   *
   * @param anno      Annotation FQN.
   * @param withTests Whether to include annotations located in test scope.
   * @return Custom annotation types.
   * @since 16
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno, final boolean withTests) {
    return getCustomMetaAnnotations(anno, withTests, true);
  }

  /**
   * Returns all custom meta annotations in defined scope, optionally filtering custom JAM implementations.
   *
   * @param anno                           Annotation FQN.
   * @param withTests                      Whether to include annotations located in test scope.
   * @param filterCustomJamImplementations Whether to filter custom JAM implementations.
   * @return Custom annotation types.
   * @see JamCustomImplementationBean
   * @since 16
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno,
                                                                              final boolean withTests,
                                                                              final boolean filterCustomJamImplementations) {
    return new Function<Module, Collection<String>>() {

      @Override
      public Collection<String> fun(final Module module) {
        if (module == null) return Collections.emptySet();

        Collection<PsiClass> psiClasses = getAnnotationTypes(module, anno);

        final Set<String> customMetaFQNs = getJamCustomFQNs(module);

        return ContainerUtil.mapNotNull(psiClasses, psiClass -> {
          String qualifiedName = psiClass.getQualifiedName();
          if (anno.equals(qualifiedName)) return null;

          if (customMetaFQNs.contains(qualifiedName)) {
            return null;
          }
          return qualifiedName;
        });
      }

      @NotNull
      private Set<String> getJamCustomFQNs(Module module) {
        if (filterCustomJamImplementations && ourJamCustomImplementationBeans.containsKey(anno)) {
          final Set<String> customMetaFQNs = new SmartHashSet<>();

          final Collection<JamCustomImplementationBean> beans = ourJamCustomImplementationBeans.get(anno);
          for (JamCustomImplementationBean bean : beans) {
            final String customMetaAnnotationFqn = bean.customMetaAnnotationFqn;
            customMetaFQNs.add(customMetaAnnotationFqn);

            Collection<PsiClass> customMetaClasses = getAnnotationTypes(module, customMetaAnnotationFqn);
            for (PsiClass customMetaClass : customMetaClasses) {
              ContainerUtil.addIfNotNull(customMetaFQNs, customMetaClass.getQualifiedName());
            }
          }
          return customMetaFQNs;
        }

        return Collections.emptySet();
      }

      private Collection<PsiClass> getAnnotationTypes(Module module, String anno) {
        return withTests ?
               JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildrenIncludingTests(anno) :
               JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(anno);
      }
    };
  }
}
