/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.profile;

import com.intellij.ide.presentation.Presentation;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashSet;
import java.util.Set;

@Presentation(typeName = "@Profile")
public class SpringJamProfile extends CommonModelElement.PsiBase implements SpringContextProfile {

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiMember myPsiMember;

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = CONTEXT_PROFILE_JAM_ANNOTATION_KEY.subKey("SpringJamProfile");
  private static final SemKey<SpringJamProfile> JAM_KEY = SpringContextProfile.CONTEXT_PROFILE_JAM_KEY.subKey("SpringJamProfile");

  public static final JamMemberMeta<PsiMember, SpringJamProfile> META = new JamMemberMeta<>(null, SpringJamProfile.class, JAM_KEY);
  private static final JamStringAttributeMeta.Collection<String> VALUE_ATTR_META = JamAttributeMeta.collectionString("value");
  private static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype().addAttribute(VALUE_ATTR_META);

  public static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.PROFILE, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    META.addAnnotation(ANNO_META);
  }

  public SpringJamProfile(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringJamProfile(@NotNull PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }


  @NotNull
  public Set<String> getNames() {
    Set<String> profiles = new LinkedHashSet<>();
    for (JamStringAttributeElement<String> element : ANNO_META.getAttribute(myPsiMember, VALUE_ATTR_META)) {
      final String value = element.getValue();
      if (StringUtil.isNotEmpty(value)) {
        for (String profile : StringUtil.split(value, ",")) {
          profiles.add(profile.trim());
        }
      }
    }
    return profiles;
  }

  @NotNull
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }
}