/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils

import com.intellij.openapi.progress.ProgressManager
import com.intellij.spring.model.CommonSpringBean
import com.intellij.spring.model.SpringBeanPointer
import com.intellij.util.Processor
import com.intellij.util.SmartList
import com.intellij.util.containers.ContainerUtil
import com.intellij.util.containers.FactoryMap

/**
 * Processor which collects [SpringBeanPointer]s ignoring beans from "ComponentScan" if they were explicitly redefined
 * in Java or XML configuration.
 * @since 2017.3
 */
class ExplicitRedefinitionAwareBeansCollector<T : CommonSpringBean> : Processor<SpringBeanPointer<T>> {

  private val beansByNameMap = FactoryMap.createMap<String, MutableList<SpringBeanPointer<T>>>(
    { _ -> SmartList() },
    { ContainerUtil.newLinkedHashMap() }
  )

  val result: Set<SpringBeanPointer<T>>
    get() = beansByNameMap.values.flatMapTo(ContainerUtil.newLinkedHashSet<SpringBeanPointer<T>>(), { mixedBeans ->
      mixedBeans.takeIf { it.size <= 1 } ?:
      mixedBeans.groupBy { it.beanClass?.qualifiedName ?: "" }.values.flatMap { mixedBeansOfOneType ->
        mixedBeansOfOneType.takeIf { it.size <= 1 } ?:
        mixedBeansOfOneType.filter { isExplicitlyDefined(it) }.takeIf { it.isNotEmpty() } ?:
        mixedBeansOfOneType
      }
    })

  override fun process(pointer: SpringBeanPointer<T>): Boolean {
    ProgressManager.checkCanceled()
    beansByNameMap.getValue(pointer.name ?: "").add(pointer)
    return true
  }

  private fun isExplicitlyDefined(pointer: SpringBeanPointer<T>) = pointer.psiElement?.isPhysical ?: false
}