/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.values.converters.resources;

import com.intellij.openapi.util.Condition;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReferenceSet;
import org.jetbrains.annotations.NotNull;

public class ResourceTypeCondition implements Condition<PsiFileSystemItem> {
  private final String[] myFileExtensions;

  public ResourceTypeCondition(@NotNull String... fileExtensions) {
    myFileExtensions = fileExtensions;
  }

  public boolean value(PsiFileSystemItem psiFileSystemItem) {
    for (String fileExtension : myFileExtensions) {
      if (hasExtension(psiFileSystemItem, fileExtension)) return true;
    }
    return false;
  }

  private static boolean hasExtension(PsiFileSystemItem psiFileSystemItem, String fileExtension) {
    if (FileReferenceSet.DIRECTORY_FILTER.value(psiFileSystemItem)) return true;
    VirtualFile virtualFile = psiFileSystemItem.getVirtualFile();

    return virtualFile != null && fileExtension.equals(virtualFile.getExtension());
  }

  @NotNull
  public String[] getExpectedExtensions() {
    return myFileExtensions;
  }
}
