/*
 * Copyright (c) 2004 JetBrains s.r.o. All  Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * -Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *
 * -Redistribution in binary form must reproduct the above copyright
 *  notice, this list of conditions and the following disclaimer in
 *  the documentation and/or other materials provided with the distribution.
 *
 * Neither the name of JetBrains or IntelliJ IDEA
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
 * ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. JETBRAINS AND ITS LICENSORS SHALL NOT
 * BE LIABLE FOR ANY DAMAGES OR LIABILITIES SUFFERED BY LICENSEE AS A RESULT
 * OF OR RELATING TO USE, MODIFICATION OR DISTRIBUTION OF THE SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL JETBRAINS OR ITS LICENSORS BE LIABLE FOR ANY LOST
 * REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL,
 * INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY
 * OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE, EVEN
 * IF JETBRAINS HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 */
package com.intellij.psi.css;

import com.intellij.icons.AllIcons;
import com.intellij.lang.css.CSSLanguage;
import com.intellij.openapi.fileTypes.LanguageFileType;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.css.descriptor.CssContextType;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

public class CssFileType extends LanguageFileType {

  @NonNls public static final String DEFAULT_EXTENSION = "css";
  public static final LanguageFileType INSTANCE = new CssFileType();
  @NonNls private static final String DESCRIPTION = "Cascading style sheet";

  public CssFileType() {
    super(CSSLanguage.INSTANCE);
  }

  @NotNull
  public String getName() {
    return "CSS";
  }

  @NotNull
  public String getDescription() {
    return DESCRIPTION;
  }

  @NotNull
  public String getDefaultExtension() {
    return DEFAULT_EXTENSION;
  }

  public Icon getIcon() {
    return AllIcons.FileTypes.Css;
  }

  @Override
  public String getCharset(@NotNull VirtualFile file, @NotNull byte[] content) {
    Charset charset = getCharsetFromCssContent(content);
    return charset != null ? charset.name() : null;
  }

  @Override
  public Charset extractCharsetFromFileContent(@Nullable Project project, @Nullable VirtualFile file, @NotNull CharSequence content) {
    return getCharsetFromCssContent(content);
  }

  public static Charset getCharsetFromCssContent(byte[] content) {
    final Charset fromBOM = CharsetToolkit.guessFromBOM(content);
    if (fromBOM != null) {
      return fromBOM;
    }
    
    String strContent;
    try {
      strContent = new String(content, "ISO-8859-1");
    }
    catch (UnsupportedEncodingException e) {
      return null;
    }

    return getCharsetFromCssContent(strContent);
  }

  @Nullable
  public static Charset getCharsetFromCssContent(@NotNull CharSequence content) {
    int charsetKeywordLength = CssContextType.CHARSET.toString().length();
    if (StringUtil.startsWith(content, CssContextType.CHARSET.toString()) && content.length() > charsetKeywordLength) {
      int i = charsetKeywordLength;
      char ch = content.charAt(i);
      while (i < content.length() && Character.isWhitespace(ch = content.charAt(i))) {
        i++;
      }
      if (ch != '"') {
        return null;
      }

      i++;
      if (i >= content.length()) {
        return null;
      }
      
      ch = content.charAt(i);
      String name = "";
      while (i < content.length() && !Character.isWhitespace(ch) && ch != '"') {
        name += ch;
        i++;
        if (i + 1 < content.length()) {
          ch = content.charAt(i);
        }
      }

      if (ch != '"') { //unclosed charset string
        return null;
      }

      return CharsetToolkit.forName(name);
    }
    return null;
  }
}
