/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.model.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.*;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.javaee.web.CommonServlet;
import com.intellij.javaee.web.CommonServletMapping;
import com.intellij.javaee.web.WebCommonClassNames;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementRef;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.NameValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * @author peter
 */
public abstract class JamServlet extends CommonModelElement.PsiBase implements CommonServlet, CommonServletMapping<CommonServlet>, JamElement {

  public static final JamStringAttributeMeta.Single<String> NAME_META = JamAttributeMeta.singleString("name");
  public static final JamStringAttributeMeta.Collection<String> URL_PATTERNS_META = JamAttributeMeta.collectionString("urlPatterns");
  public static final JamStringAttributeMeta.Collection<String> VALUES_META = JamAttributeMeta.collectionString("value");

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.INIT_PARAM_ANNO_META, JamInitParam.class);

  @NonNls public static final String ANNO_NAME = WebCommonClassNames.ANNOTATION_WEB_SERVLET;

  public static final JamAnnotationMeta SERVLET_ANNO_META = new JamAnnotationMeta(ANNO_NAME).
    addAttribute(NAME_META).
    addAttribute(URL_PATTERNS_META).
    addAttribute(VALUES_META).
    addAttribute(INIT_PARAMS_META);

  public static final JamClassMeta<JamServlet> SERVLET_CLASS_META = new JamClassMeta<>(JamServlet.class);

  @JamPsiConnector
  public abstract PsiClass getPsiClass();

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return getPsiClass();
  }

  @Override
  @JamPsiValidity
  public abstract boolean isValid();

  @NameValue
  public JamStringAttributeElement<String> getServletName() {
    return SERVLET_ANNO_META.getAttribute(getPsiClass(), NAME_META);
  }

  public List<JamStringAttributeElement<String>> getUrlPatterns() {
    return ContainerUtil.concat(SERVLET_ANNO_META.getAttribute(getPsiClass(), URL_PATTERNS_META),
                                SERVLET_ANNO_META.getAttribute(getPsiClass(), VALUES_META));
  }

  public List<JamInitParam> getInitParams() {
    return SERVLET_ANNO_META.getAttribute(getPsiClass(), INIT_PARAMS_META);
  }

  public JamInitParam addInitParam() {
    return INIT_PARAMS_META.addAttribute(PsiElementRef.real(SERVLET_ANNO_META.getAnnotation(getPsiClass())));
  }

  public CommonServlet getServlet() {
    return this;
  }

  public PsiElement getMappingElement() {
    return SERVLET_ANNO_META.getAnnotation(getPsiClass());
  }
}
