/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.jam

import com.intellij.codeInsight.completion.CompletionUtil
import com.intellij.jam.reflect.JamStringAttributeMeta
import com.intellij.patterns.PsiJavaElementPattern
import com.intellij.patterns.PsiJavaPatterns
import com.intellij.patterns.PsiJavaPatterns.psiLiteral
import com.intellij.patterns.PsiJavaPatterns.psiNameValuePair
import com.intellij.psi.*
import com.intellij.semantic.SemService
import org.jetbrains.uast.*

class JamReferenceContributor : PsiReferenceContributor() {

  private fun retrievePsiAnnotation(uElement: UElement?): PsiAnnotation? =
    (uElement?.getParentOfType<UAnnotation>(true) as? JvmDeclarationUElement)?.javaPsi as? PsiAnnotation

  override fun registerReferenceProviders(registrar: PsiReferenceRegistrar) {

    registrar.registerUastReferenceProvider(
      { element, _ -> element is ULiteralExpression && element.getParentOfType<UAnnotation>() != null },
      uastLiteralReferenceProvider(
        fun(uElement: ULiteralExpression, physicalPsiHost: PsiLanguageInjectionHost): Array<PsiReference> {
          val psiAnnotation = retrievePsiAnnotation(uElement) ?: return PsiReference.EMPTY_ARRAY
          val originalPsiAnnotation = retrievePsiAnnotation(CompletionUtil.getOriginalOrSelf(physicalPsiHost).toUElement())
                                      ?: return PsiReference.EMPTY_ARRAY
          val metas = SemService.getSemService(originalPsiAnnotation.project).getSemElements(JamService.ANNO_META_KEY,
                                                                                             originalPsiAnnotation)
          val namedExpression = uElement.getParentOfType<UNamedExpression>(true) ?: return PsiReference.EMPTY_ARRAY
          for (annotationMeta in metas) {
            val meta = annotationMeta?.findAttribute(namedExpression.name) as? JamStringAttributeMeta<*, *> ?: continue
            val converter = meta.converter as JamConverter<in Any>
            val jam = meta.getJam(PsiElementRef.real(psiAnnotation))

            (jam  as? List<JamStringAttributeElement<*>>)?.let { list ->
              return list.filter { isJamElementFromHost(it, physicalPsiHost) }.flatMap {
                converter.createReferences(it).toList().map { wrapRef(it, physicalPsiHost) }
              }.toTypedArray()
            }
            (jam as? JamStringAttributeElement<*>)?.let {
              return converter.createReferences(it).map {
                wrapRef(it, physicalPsiHost)
              }.toTypedArray()
            }

          }

          return PsiReference.EMPTY_ARRAY
        }
      )
    )

  }

  private fun isJamElementFromHost(attributeElement: JamStringAttributeElement<*>, physicalPsiHost: PsiLanguageInjectionHost) =
    attributeElement.psiElement == physicalPsiHost ||
    (attributeElement.psiElement.toUElement() as? ULiteralExpression)?.psiLanguageInjectionHost == physicalPsiHost

  companion object {
    private val NAME_VALUE_PAIR = psiNameValuePair().withParent(PsiAnnotationParameterList::class.java)

    @JvmField
    val STRING_IN_ANNO: PsiJavaElementPattern.Capture<PsiLiteral> = psiLiteral().withParent(
      PsiJavaPatterns.or(NAME_VALUE_PAIR, PsiJavaPatterns.psiElement(PsiArrayInitializerMemberValue::class.java).withParent(NAME_VALUE_PAIR)
      ))
  }
}

private fun wrapRef(it: PsiReference, physicalPsi: PsiElement): PsiReference =
  if (it.element === physicalPsi) it else NonPhysicalReferenceWrapper(physicalPsi, it)

class NonPhysicalReferenceWrapper(physicalElement: PsiElement, val wrappedReference: PsiReference) :
  PsiReferenceBase<PsiElement>(physicalElement, wrappedReference.rangeInElement) {
  override fun resolve(): PsiElement? = wrappedReference.resolve()

  override fun getVariants(): Array<Any> = wrappedReference.variants

  override fun isSoft(): Boolean = wrappedReference.isSoft
}
