/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.RecursionManager;
import com.intellij.openapi.util.VolatileNotNullLazyValue;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.util.Processor;
import com.intellij.util.Processors;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.*;

public abstract class CachedLocalModel implements CommonSpringModel {

  private volatile Collection<SpringBeanPointer> myLocalBeans; // beans defined in current file

  private final VolatileNotNullLazyValue<Collection<SpringBeanPointer>> myLocalDomBeans =
    new VolatileNotNullLazyValue<Collection<SpringBeanPointer>>() {
      @NotNull
      protected Collection<SpringBeanPointer> compute() {
        return calculateDomBeans();
      }
    };

  @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
  private final Map<SpringQualifier, List<SpringBeanPointer>> myLocalBeansByQualifier =
    ConcurrentFactoryMap.createMap(key -> computeLocalBeansByQualifier(key));

  @NotNull
  public final Collection<SpringBeanPointer> getAllCommonBeans() {
    final Set<SpringBeanPointer> pointers = ContainerUtil.newLinkedHashSet();
    processAllBeans(Processors.cancelableCollectProcessor(pointers));
    return Collections.unmodifiableSet(pointers);
  }

  @Override
  public boolean processAllBeans(@NotNull Processor<SpringBeanPointer> processor) {
    return processAllBeansRelatedModels(getDelegateProcessor(processor));
  }

  public Collection<SpringBeanPointer> getLocalBeans() {
    return myLocalBeans == null ? myLocalBeans = calculateLocalBeans() : myLocalBeans;
  }

  @NotNull
  public final Collection<SpringBeanPointer> getAllDomBeans() {
    return myLocalDomBeans.getValue();
  }

  /**
   * All related models: imported, component-scan, implicit, etc. (these models could be cached with other dependencies)
   */
  protected Iterable<CommonSpringModel> getRelatedModels() {
    return Collections.emptySet();
  }

  protected abstract Collection<SpringBeanPointer> calculateLocalBeans();

  protected abstract Collection<SpringBeanPointer> calculateDomBeans();

  @NotNull
  public List<SpringBeanPointer> findQualifiedBeans(@NotNull final SpringQualifier qualifier) {
    return myLocalBeansByQualifier.get(qualifier);
  }

  private List<SpringBeanPointer> computeLocalBeansByQualifier(final SpringQualifier springQualifier) {
    final List<SpringBeanPointer> beans = new SmartList<>();
    final Collection<SpringBeanPointer> pointers = getAllCommonBeans();
    for (SpringBeanPointer beanPointer : pointers) {
      if (!beanPointer.isValid()) continue;
      final CommonSpringBean bean = beanPointer.getSpringBean();
      for (SpringQualifier qualifier : bean.getSpringQualifiers()) {
        if (qualifier.compareQualifiers(springQualifier, getModule())) {
          beans.add(beanPointer);
        }
      }
    }
    return beans.isEmpty() ? Collections.emptyList() : beans;
  }

  public boolean processRelatedModels(@NotNull final Processor<CommonSpringModel> processor) {
    final Boolean aBoolean = RecursionManager.doPreventingRecursion(this, false, () -> {
      for (CommonSpringModel model : getRelatedModels()) {
        ProgressManager.checkCanceled();
        if (!processor.process(model)) return Boolean.FALSE;
      }
      return Boolean.TRUE;
    });

    return aBoolean == null ? true : aBoolean;
  }

  @Override
  public boolean processByClass(@NotNull final SpringModelSearchParameters.BeanClass params,
                                @NotNull final Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    return processRelatedModels(params, getDelegateProcessor(processor));
  }

  @Override
  public boolean processByName(@NotNull final SpringModelSearchParameters.BeanName params,
                               @NotNull final Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    return processRelatedModels(params, getDelegateProcessor(processor));
  }

  protected boolean processAllBeansRelatedModels(final Processor<SpringBeanPointer> processor) {
    return processRelatedModels(model -> !shouldProcess(model, processor) || model.processAllBeans(processor));
  }

  protected boolean processRelatedModels(final SpringModelSearchParameters.BeanClass params, final Processor<SpringBeanPointer> processor) {
    return processRelatedModels(model -> !shouldProcess(model, processor) || model.processByClass(params, processor));
  }

  protected boolean processRelatedModels(final SpringModelSearchParameters.BeanName params, final Processor<SpringBeanPointer> processor) {
    return processRelatedModels(model -> !shouldProcess(model, processor) || model.processByName(params, processor));
  }

  private static boolean shouldProcess(@NotNull CommonSpringModel model, @NotNull Processor<SpringBeanPointer> processor) {
    if (processor instanceof VisitedModelsDelegateProcessor) {
      if (((VisitedModelsDelegateProcessor)processor).hasBeenVisited(model)) return false;
      ((VisitedModelsDelegateProcessor)processor).addVisited(model);
    }
    return true;
  }

  @NotNull
  private static VisitedModelsDelegateProcessor getDelegateProcessor(@NotNull Processor<SpringBeanPointer> processor) {
    if (processor instanceof VisitedModelsDelegateProcessor) {
      return (VisitedModelsDelegateProcessor)processor;
    }
    return new VisitedModelsDelegateProcessor(processor);
  }
}
