/*
 * Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
 */
package com.intellij.spring.gutter.groups;

import com.intellij.codeInsight.daemon.GutterIconNavigationHandler;
import com.intellij.codeInsight.daemon.RelatedItemLineMarkerInfo;
import com.intellij.codeInsight.navigation.NavigationGutterIconBuilder;
import com.intellij.codeInsight.navigation.NavigationGutterIconRenderer;
import com.intellij.navigation.GotoRelatedItem;
import com.intellij.openapi.editor.markup.GutterIconRenderer;
import com.intellij.psi.PsiElement;
import com.intellij.util.NotNullFunction;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Collection;

public class SpringGutterIconBuilder<T> extends NavigationGutterIconBuilder<T> {
  @NonNls public static final GutterIconRenderer.Alignment DEFAULT_GUTTER_ICON_ALIGNMENT = GutterIconRenderer.Alignment.LEFT;

  @Nullable private GutterIconNavigationHandler<PsiElement> myNavigationHandler;

  public static SpringGutterIconBuilder<PsiElement> createBuilder(@NotNull Icon icon) {
    return new SpringGutterIconBuilder<>(icon, DEFAULT_PSI_CONVERTOR, PSI_GOTO_RELATED_ITEM_PROVIDER);
  }

  public static <T> SpringGutterIconBuilder<T> createBuilder(@NotNull Icon icon,
                                                             @NotNull NotNullFunction<T, Collection<? extends PsiElement>> converter,
                                                             @Nullable NotNullFunction<T, Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
    return new SpringGutterIconBuilder<>(icon, converter, gotoRelatedItemProvider);
  }

  private SpringGutterIconBuilder(@NotNull Icon icon,
                                  @NotNull NotNullFunction<T, Collection<? extends PsiElement>> converter,
                                  @Nullable NotNullFunction<T, Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
    super(icon, converter, gotoRelatedItemProvider);
  }

  public SpringGutterIconBuilder<T> setNavigationHandler(@Nullable GutterIconNavigationHandler<PsiElement> navigationHandler) {
    myNavigationHandler = navigationHandler;
    return this;
  }

  @Nullable
  private GutterIconNavigationHandler<PsiElement> createNavigationHandler(@NotNull NavigationGutterIconRenderer renderer) {
    return myNavigationHandler == null && renderer.isNavigateAction() ? renderer : myNavigationHandler;
  }

  @NotNull
  public RelatedItemLineMarkerInfo<PsiElement> createSpringGroupLineMarkerInfo(@NotNull PsiElement element) {
    NavigationGutterIconRenderer renderer = createGutterIconRenderer(element.getProject());
    return new SpringGroupRelatedItemLineMarkerInfo(element, renderer, createNavigationHandler(renderer), getGotoTargets());
  }

  @NotNull
  public RelatedItemLineMarkerInfo<PsiElement> createSpringRelatedMergeableLineMarkerInfo(@NotNull PsiElement element) {
    NavigationGutterIconRenderer renderer = createGutterIconRenderer(element.getProject());
    return new SpringRelatedMergeableLineMarkerInfo(element, renderer, createNavigationHandler(renderer), getGotoTargets());
  }
}
