/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiAnnotationMemberValue;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.converters.SpringConverterUtil;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.utils.SpringModelUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * @since 14
 */
public class SpringBeanReferenceJamConverter extends JamSimpleReferenceConverter<SpringBeanPointer> {

  @Nullable
  private final String myBaseClass;

  /**
   * @param baseClass Required base class or {@code null} (provide custom {@link #getVariants(JamStringAttributeElement)}).
   */
  public SpringBeanReferenceJamConverter(@Nullable String baseClass) {
    myBaseClass = baseClass;
  }

  /**
   * @return Required base class or {@code null}.
   * @since 2018.1
   */
  @Nullable
  public String getBaseClass() {
    return myBaseClass;
  }

  @Override
  public SpringBeanPointer fromString(@Nullable String s, JamStringAttributeElement<SpringBeanPointer> context) {
    if (StringUtil.isEmptyOrSpaces(s)) {
      return null;
    }

    final CommonSpringModel model = getSpringModel(context.getPsiElement());
    return SpringModelSearchers.findBean(model, s);
  }

  @Override
  public Collection<SpringBeanPointer> getVariants(JamStringAttributeElement<SpringBeanPointer> context) {
    final PsiAnnotationMemberValue psiElement = context.getPsiElement();
    if (psiElement == null) return Collections.emptyList();

    if (myBaseClass == null) {
      final CommonSpringModel model = getSpringModel(psiElement);
      return model.getAllCommonBeans();
    }

    final PsiClass psiClass = JavaPsiFacade.getInstance(psiElement.getProject())
      .findClass(myBaseClass, psiElement.getResolveScope());
    if (psiClass == null) return Collections.emptyList();

    final CommonSpringModel model = getSpringModel(psiElement);
    final SpringModelSearchParameters.BeanClass searchParameters =
      SpringModelSearchParameters.byClass(psiClass).withInheritors().effectiveBeanTypes();
    return SpringModelSearchers.findBeans(model, searchParameters);
  }

  @NotNull
  @Override
  protected LookupElement createLookupElementFor(@NotNull SpringBeanPointer target) {
    final LookupElement variant = SpringConverterUtil.createCompletionVariant(target);
    return variant != null ? variant : super.createLookupElementFor(target);
  }

  @Override
  protected PsiElement getPsiElementFor(@NotNull SpringBeanPointer target) {
    return target.getSpringBean().getIdentifyingPsiElement();
  }

  /**
   * Spring Model for converter.
   *
   * @param psiElement Current element.
   * @return Spring Model for resolving/completion.
   * @since 15
   */
  @NotNull
  protected CommonSpringModel getSpringModel(PsiElement psiElement) {
    return SpringModelUtils.getInstance().getSpringModel(psiElement);
  }
}
