/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.xml;

import com.intellij.openapi.module.Module;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.factories.CustomFactoryMethodTypeHandler;
import com.intellij.spring.factories.SpringFactoryBeansManager;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.beans.*;
import com.intellij.util.xml.*;
import com.intellij.util.xml.converters.values.ClassValueConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import static com.intellij.psi.util.CachedValueProvider.Result;

/**
 * @author peter
 */
public abstract class AbstractDomSpringBean implements CommonSpringBean {
  private CachedValue<PsiClass> myClassCachedValue;
  private CachedValue<PsiClass> myFactoriesClassCachedValue;

  @Nullable
  @Override
  @NameValue
  public abstract String getBeanName();

  @Nullable
  public GenericValue<PsiMethod> getFactoryMethod() {
    return null;
  }

  @Nullable
  public GenericValue<SpringBeanPointer> getFactoryBean() {
    return null;
  }

  @Nullable
  public abstract String getClassName();

  @Nullable
  public final PsiType getBeanType() {
    final PsiClass beanClass = getBeanClass(null, true);
    return beanClass != null ? PsiTypesUtil.getClassType(beanClass) : null;
  }

  @Nullable
  public PsiType getBeanType(boolean considerFactories) {
    final PsiClass beanClass = getBeanClass(null, considerFactories);
    return beanClass != null ? PsiTypesUtil.getClassType(beanClass) : null;
  }

  @Nullable
  public abstract Module getModule();

  @NotNull
  public abstract PsiFile getContainingFile();

  @Nullable
  public PsiClass getBeanClass(@Nullable Set<AbstractDomSpringBean> visited, boolean considerFactories) {
    if (visited != null && visited.contains(this)) return null;

    if (considerFactories) {
      if (myFactoriesClassCachedValue == null) {
        myFactoriesClassCachedValue = CachedValuesManager.getManager(getContainingFile().getProject())
          .createCachedValue(() -> {
            PsiClass beanClass = calculateFactoriesBeanClass(this);
            return Result.createSingleDependency(beanClass, PsiModificationTracker.JAVA_STRUCTURE_MODIFICATION_COUNT);
          });
      }
      return myFactoriesClassCachedValue.getValue();
    }

    return getClassAttributeValue();
  }

  public PsiClass getClassAttributeValue() {
    if (myClassCachedValue == null) {
      myClassCachedValue = CachedValuesManager.getManager(getContainingFile().getProject())
        .createCachedValue(() -> {
          PsiClass beanClass = calculateBeanClass(this);
          return Result.createSingleDependency(beanClass, PsiModificationTracker.JAVA_STRUCTURE_MODIFICATION_COUNT);
        });
    }
    return myClassCachedValue.getValue();
  }


  @Nullable
  private static PsiClass calculateFactoriesBeanClass(@NotNull AbstractDomSpringBean springBean) {
    final GenericValue<PsiMethod> value = springBean.getFactoryMethod();
    final PsiMethod factoryMethod = value == null ? null : value.getValue();
    if (factoryMethod == null) {
      return calculateBeanClass(springBean);
    }

    final GenericValue<SpringBeanPointer> factoryBean = springBean.getFactoryBean();
    if (!SpringFactoryBeansManager.getInstance()
      .isValidFactoryMethod(factoryMethod, factoryBean != null && factoryBean.getValue() != null)) {
      return calculateBeanClass(springBean);
    }

    for (CustomFactoryMethodTypeHandler typeHandler : CustomFactoryMethodTypeHandler.EP_NAME.getExtensions()) {
      PsiType factoryMethodType = typeHandler.getFactoryMethodType(factoryMethod, springBean);
      if (factoryMethodType instanceof PsiClassType) return ((PsiClassType)factoryMethodType).resolve();
    }

    final PsiType returnType = factoryMethod.getReturnType();
    if (returnType instanceof PsiClassType) {
      PsiTypeParameter[] typeParameters = factoryMethod.getTypeParameters();
      PsiClass resolve = ((PsiClassType)returnType).resolve();
      if (resolve == null) return null;

      if (!(springBean instanceof SpringBean)) return resolve;

      for (PsiTypeParameter typeParameter : typeParameters) {
        if (typeParameter.equals(resolve)) {
          // trying to find Class<T> parameter
          String text = "java.lang.Class<" + typeParameter.getName() + ">";
          PsiParameter[] parameters = factoryMethod.getParameterList().getParameters();
          for (int i = 0, length = parameters.length; i < length; i++) {
            final PsiParameter psiParameter = parameters[i];
            PsiType type = psiParameter.getType();
            if (type.getCanonicalText().equals(text)) {
              ConstructorArgumentValues values = new ConstructorArgumentValues();
              values.init((SpringBean)springBean);
              ConstructorArgDefinition definition = values.resolve(i, psiParameter, null);
              if (definition != null) {
                final GenericDomValue<?> valueElement = definition.getValueElement();
                String rawText = valueElement == null ? null : valueElement.getRawText();
                if (rawText != null) {
                  return DomJavaUtil.findClass(rawText, springBean.getContainingFile(), springBean.getModule(), null);
                }
              }
            }
          }
          return resolve;
        }
      }
      return resolve;
    }


    return calculateBeanClass(springBean);
  }

  @Nullable
  private static PsiClass calculateBeanClass(@NotNull AbstractDomSpringBean springBean) {
    String className = springBean.getClassName();
    if (className == null) {
      return null;
    }

    final PsiFile containingFile = springBean.getContainingFile();
    final GlobalSearchScope scope = ClassValueConverter.getScope(containingFile.getProject(), springBean.getModule(), containingFile);
    return DomJavaUtil.findClass(className.trim(), containingFile, null, scope);
  }

  @NotNull
  public Collection<SpringQualifier> getSpringQualifiers() {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public SpringProfile getProfile() {
    final Beans beans = getBeansParent();

    if (beans != null) {
      final SpringDomProfile profile = beans.getProfile();

      if (DomUtil.hasXml(profile)) return profile;
    }
    return SpringProfile.DEFAULT;
  }

  @Nullable
  public abstract Beans getBeansParent();

  @Override
  public boolean isPrimary() {
    return false;
  }
}
