/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.toolWindow;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.module.ModuleType;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ui.configuration.ModulesAlphaComparator;
import com.intellij.openapi.roots.ui.configuration.ModulesConfigurator;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.spring.facet.SpringFileSetService;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.settings.SpringGeneralSettings;
import com.intellij.ui.FinderRecursivePanel;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public abstract class SpringModulesPanelBase extends FinderRecursivePanel<Module> {

  protected SpringModulesPanelBase(Project project, String groupId) {
    super(project, groupId);
  }

  @NotNull
  @Override
  public List<Module> getListItems() {
    boolean autoConfigurationMode = SpringGeneralSettings.getInstance(getProject()).isAllowAutoConfigurationMode();

    if (!SpringCommonUtils.hasSpringFacets(getProject()) && !autoConfigurationMode) {
      return Collections.emptyList();
    }

    List<Module> items = new ArrayList<>();
    Module[] modules = ModuleManager.getInstance(getProject()).getModules();
    for (Module module : modules) {
      if (autoConfigurationMode || SpringCommonUtils.hasSpringFacet(module)) {
        items.add(module);
      }
    }
    Collections.sort(items, ModulesAlphaComparator.INSTANCE);
    return items;
  }

  @NotNull
  @Override
  public String getItemText(Module module) {
    return module.getName();
  }

  @Nullable
  @Override
  public Icon getItemIcon(Module module) {
    return module.isDisposed()? null : ModuleType.get(module).getIcon();
  }

  @Override
  public boolean performEditAction() {
    final Module module = getSelectedValue();
    assert module != null;
    ModulesConfigurator.showDialog(getProject(), module.getName(), null);
    return true;
  }

  @Override
  public boolean hasChildren(Module module) {
    if (module.isDisposed()) return false;
    if (DumbService.isDumb(getProject())) return false;

    final SpringFacet springFacet = SpringFacet.getInstance(module);
    if (springFacet == null) return false;
    return !SpringFileSetService.getInstance().getAllSets(springFacet).isEmpty();
  }
}