// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application;

import com.intellij.jam.JamService;
import com.intellij.openapi.application.ReadAction;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiModifier;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiMethodUtil;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.spring.boot.model.autoconfigure.SpringBootAutoconfigureClassesConstants;
import com.intellij.spring.boot.model.autoconfigure.jam.EnableAutoConfiguration;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.uast.UClass;
import org.jetbrains.uast.UFile;
import org.jetbrains.uast.UMethod;
import org.jetbrains.uast.UastContextKt;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class SpringBootApplicationUtil {

  private static final Condition<PsiClass> RUNNABLE_CONDITION =
    psiClass -> PsiMethodUtil.MAIN_CLASS.value(psiClass) && psiClass.hasModifierProperty(PsiModifier.PUBLIC);

  public static boolean isSpringApplication(PsiClass psiClass) {
    if (!RUNNABLE_CONDITION.value(psiClass)) {
      return false;
    }

    final EnableAutoConfiguration element =
      JamService.getJamService(psiClass.getProject()).getJamElement(psiClass, EnableAutoConfiguration.META);
    return element != null;
  }

  public static List<PsiClass> getSpringApplications(final Module module) {
    if (DumbService.isDumb(module.getProject())) return Collections.emptyList();

    if (SpringCommonUtils.findLibraryClass(module, SpringBootAutoconfigureClassesConstants.ENABLE_AUTO_CONFIGURATION) == null) {
      return Collections.emptyList();
    }

    return CachedValuesManager.getManager(module.getProject()).getCachedValue(module, () -> {
      final JamService jamService = JamService.getJamService(module.getProject());
      final GlobalSearchScope scope = module.getModuleScope(false);

      final Collection<String> fqns = new ArrayList<>(EnableAutoConfiguration.getAnnotations().fun(module));
      fqns.add(SpringBootAutoconfigureClassesConstants.ENABLE_AUTO_CONFIGURATION);

      List<PsiClass> applications = new SmartList<>();
      for (String fqn : fqns) {
        final List<EnableAutoConfiguration> elements = jamService.getJamClassElements(EnableAutoConfiguration.JAM_KEY, fqn, scope);
        for (EnableAutoConfiguration element : elements) {
          final PsiClass annotatedClass = element.getPsiElement();
          if (RUNNABLE_CONDITION.value(annotatedClass)) {
            applications.add(annotatedClass);
          }
        }
      }

      return CachedValueProvider.Result.create(applications, PsiModificationTracker.JAVA_STRUCTURE_MODIFICATION_COUNT);
    });
  }

  public static boolean hasMainMethod(PsiClass psiClass) {
    return PsiMethodUtil.hasMainMethod(guessUastMainClass(psiClass));
  }

  @Nullable
  public static PsiClass guessUastMainClass(@Nullable PsiClass springBootClass) {
    if (springBootClass == null) return null;
    //It is a workaround for KT-18054, ideally it should be done as `UastContextKt.toUElement(app.containingFile,UFile::class).getClasses()`
    PsiFile langFile = ReadAction.compute(() -> {
      VirtualFile virtualFile = springBootClass.getContainingFile().getVirtualFile();
      if (virtualFile == null) return null;
      return springBootClass.getManager().findFile(virtualFile);
    });
    if (langFile == null) return springBootClass;

    UFile file = UastContextKt.toUElement(langFile, UFile.class);
    if (file == null) return springBootClass;

    List<UMethod> mainMethodCandidates = new SmartList<>();
    for (UClass uClass : file.getClasses()) {
      for (UMethod method : uClass.getMethods()) {
        if (PsiMethodUtil.isMainMethod(method)) {
          mainMethodCandidates.add(method);
        }
      }
    }
    if (mainMethodCandidates.size() == 1) {
      return mainMethodCandidates.get(0).getContainingClass();
    }
    return springBootClass;
  }
}
