// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.ProblemHighlightType;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReference;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey;
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey.Deprecation;
import com.intellij.spring.boot.application.metadata.additional.DefineLocalMetaConfigKeyFix;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.MultiMap;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;

import java.util.List;
import java.util.Set;

public class SpringBootConfigFileHighlightingUtil {

  private static final Condition<PsiReference> RELEVANT_REFERENCE_CONDITION =
    reference -> !reference.isSoft() &&
                 !(reference instanceof SpringBootPlaceholderReference);

  private final ProblemsHolder myHolder;

  public SpringBootConfigFileHighlightingUtil(ProblemsHolder holder) {
    myHolder = holder;
  }

  /**
   * Highlights all unresolved value references and class references not matching required super type(s).
   * <p/>
   * If there are multiple references per range (via multiple providers registered
   * in {@value com.intellij.spring.boot.SpringBootConfigFileConstants#ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON})
   * and at least one of them resolves, no error highlighting is performed.
   *
   * @param valueElement Value element.
   */
  public void highlightValueReferences(@NotNull PsiElement valueElement) {
    // filter soft && placeholder expressions (runtime dependent)
    final List<PsiReference> valueReferences =
      ContainerUtil.filter(valueElement.getReferences(), RELEVANT_REFERENCE_CONDITION);

    MultiMap<Integer, PsiReference> unresolvedReferences = MultiMap.createSmart();
    Set<Integer> resolvedReferencesOffsets = new SmartHashSet<>();
    for (PsiReference reference : valueReferences) {
      int startOffset = reference.getRangeInElement().getStartOffset();
      boolean unresolved = reference instanceof PsiPolyVariantReference
                           ? ((PsiPolyVariantReference)reference).multiResolve(false).length == 0
                           : reference.resolve() == null;

      if (!unresolved) {
        unresolvedReferences.remove(startOffset);
        resolvedReferencesOffsets.add(startOffset);

        if (reference instanceof JavaClassReference) {
          highlightJavaClassReferenceExtends(myHolder, (JavaClassReference)reference);
        }
      }
      else if (!resolvedReferencesOffsets.contains(startOffset)) {
        unresolvedReferences.putValue(startOffset, reference);
      }
    }

    for (PsiReference reference : unresolvedReferences.values()) {
      myHolder.registerProblem(reference, ProblemsHolder.unresolvedReferenceMessage(reference), ProblemHighlightType.ERROR);
    }
  }

  private static void highlightJavaClassReferenceExtends(ProblemsHolder holder, JavaClassReference reference) {
    final List<String> extendClassNames = reference.getSuperClasses();
    if (extendClassNames.isEmpty()) {
      return;
    }

    PsiElement resolve = reference.resolve();
    if (!(resolve instanceof PsiClass)) {
      return;
    }

    for (String extend : extendClassNames) {
      if (InheritanceUtil.isInheritor((PsiClass)resolve, extend)) {
        return;
      }
    }

    holder.registerProblem(reference.getElement(),
                           ElementManipulators.getValueText(reference.getElement()) +
                           " is not assignable to " +
                           StringUtil.join(extendClassNames, "|"),
                           ProblemHighlightType.ERROR);
  }

  public void highlightDeprecatedConfigKey(PsiElement keyElement,
                                           SpringBootApplicationMetaConfigKey configKey,
                                           LocalQuickFix... quickFixes) {
    final Deprecation deprecation = configKey.getDeprecation();
    final String reasonShortText = deprecation.getReason().getShortText();
    String reason = StringUtil.isNotEmpty(reasonShortText) ? ": " + reasonShortText :
                    " configuration property '" + configKey.getName() + "'";

    ProblemHighlightType problemHighlightType = deprecation.getLevel() == Deprecation.DeprecationLevel.ERROR
                                                ? ProblemHighlightType.GENERIC_ERROR
                                                : ProblemHighlightType.LIKE_DEPRECATED;

    myHolder.registerProblem(keyElement,
                             "Deprecated" + reason,
                             problemHighlightType,
                             quickFixes);
  }

  public void highlightUnresolvedConfigKey(PsiElement keyElement, String qualifiedConfigKeyName) {
    myHolder.registerProblem(keyElement,
                             "Cannot resolve configuration property '" + qualifiedConfigKeyName + "'",
                             ProblemHighlightType.GENERIC_ERROR_OR_WARNING,
                             new DefineLocalMetaConfigKeyFix(qualifiedConfigKeyName));
  }
}
