// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.codeInsight.documentation.DocumentationManager;
import com.intellij.codeInsight.documentation.DocumentationManagerUtil;
import com.intellij.codeInsight.javadoc.JavaDocInfoGenerator;
import com.intellij.lang.documentation.AbstractDocumentationProvider;
import com.intellij.lang.documentation.DocumentationProvider;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.beanProperties.BeanPropertyElement;
import com.intellij.psi.javadoc.PsiDocComment;
import com.intellij.psi.util.PropertyUtilBase;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.util.Function;
import com.intellij.util.ObjectUtils;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.xml.util.XmlUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Locale;

public abstract class ConfigKeyDocumentationProviderBase extends AbstractDocumentationProvider {

  private static final String CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX = "SPRING_BOOT_CONFIG_KEY_DECLARATION_PSI_ELEMENT_";
  private static final String CONFIG_KEY_LINK_SEPARATOR = "___";

  /**
   * Stores current Module when constructing {@link ConfigKeyDeclarationPsiElement}.
   */
  public static final Key<Module> CONFIG_KEY_DECLARATION_MODULE = Key.create("ConfigKeyDeclarationModule");

  @Nullable
  @Override
  public String getQuickNavigateInfo(PsiElement element, PsiElement originalElement) {
    // see ConfigKeyDeclarationPsiElement.getLanguage()
    if (element instanceof ConfigKeyDeclarationPsiElement) {
      final ConfigKeyDeclarationPsiElement configKey = (ConfigKeyDeclarationPsiElement)element;
      return "<b>" + configKey.getName() + "</b>" +
             " [" + XmlUtil.escape(configKey.getLibraryName()) + "]\n" +
             XmlUtil.escape(configKey.getLocationString());
    }

    final String valueHintDocumentation = getValueHintDocumentation(element);
    if (valueHintDocumentation != null) {
      return valueHintDocumentation;
    }

    return super.getQuickNavigateInfo(element, originalElement);
  }

  @Nullable
  private static String getValueHintDocumentation(PsiElement element) {
    if (element instanceof ValueHintPsiElement) {
      final SpringBootApplicationMetaConfigKey.ValueHint hint =
        ((ValueHintPsiElement)element).getValueHint();
      return "<b>" + hint.getValue() + "</b>: " + hint.getDescriptionText().getFullText();
    }
    return null;
  }

  @Override
  public String generateDoc(PsiElement element, @Nullable PsiElement originalElement) {
    if (element instanceof ConfigKeyDeclarationPsiElement) {
      final ConfigKeyDeclarationPsiElement configKeyDeclarationPsiElement = (ConfigKeyDeclarationPsiElement)element;
      final String keyName = configKeyDeclarationPsiElement.getName();
      return getDocumentationTextForKey(ObjectUtils.chooseNotNull(originalElement, element), keyName);
    }

    final String valueHintDocumentation = getValueHintDocumentation(element);
    if (valueHintDocumentation != null) {
      return valueHintDocumentation;
    }

    return generateDocForProperty(element);
  }

  @Override
  public PsiElement getDocumentationElementForLink(PsiManager psiManager, String link, PsiElement context) {
    if (!StringUtil.startsWith(link, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX)) {
      return super.getDocumentationElementForLink(psiManager, link, context);
    }

    // return "fake" ConfigKeyDeclarationPsiElement
    String fqn = StringUtil.substringAfter(link, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX);
    assert fqn != null : link;
    String moduleName = StringUtil.substringBefore(fqn, CONFIG_KEY_LINK_SEPARATOR);
    assert moduleName != null : link;
    String key = StringUtil.substringAfter(fqn, CONFIG_KEY_LINK_SEPARATOR);

    final Module module = ModuleManager.getInstance(psiManager.getProject()).findModuleByName(moduleName);
    final SpringBootApplicationMetaConfigKey replacementKey =
      SpringBootApplicationMetaConfigKeyManager.getInstance().findApplicationMetaConfigKey(module, key);
    if (replacementKey != null) {
      return replacementKey.getDeclaration();
    }
    return super.getDocumentationElementForLink(psiManager, link, context);
  }

  @Nullable
  protected abstract String getConfigKey(PsiElement configKeyElement);

  /**
   * Try to provide doc for "normal" property due to multi-resolve.
   */
  private String generateDocForProperty(PsiElement element) {
    String configKey = getConfigKey(element);
    if (configKey == null) {
      return generateDocForBeanProperty(element);
    }

    return getDocumentationTextForKey(element, configKey);
  }

  @Nullable
  private static String generateDocForBeanProperty(PsiElement element) {
    if (!(element instanceof BeanPropertyElement)) return null;

    BeanPropertyElement property = (BeanPropertyElement)element;
    final PsiElement originalElement = getOriginalDocumentationElement(property.getMethod());
    return getOriginalDocForPsiMember(originalElement);
  }

  @Nullable
  private static String getDocumentationTextForKey(@Nullable PsiElement originalElement,
                                                   @Nullable String keyName) {
    if (originalElement == null) {
      return null;
    }

    Module module = ObjectUtils.chooseNotNull(ModuleUtilCore.findModuleForPsiElement(originalElement),
                                              originalElement.getUserData(CONFIG_KEY_DECLARATION_MODULE));
    final SpringBootApplicationMetaConfigKey key =
      SpringBootApplicationMetaConfigKeyManager.getInstance().findApplicationMetaConfigKey(module, keyName);
    if (key == null) {
      return null;
    }

    StringBuilder sb = new StringBuilder();
    sb.append("<b>").append(keyName).append("</b><br/>");
    JavaDocInfoGenerator.generateType(sb, key.getType(), originalElement, true);
    sb.append("<br/>");

    final SpringBootApplicationMetaConfigKey.Deprecation deprecation = key.getDeprecation();
    if (deprecation != SpringBootApplicationMetaConfigKey.Deprecation.NOT_DEPRECATED) {
      if (deprecation.getLevel() == SpringBootApplicationMetaConfigKey.Deprecation.DeprecationLevel.ERROR) {
        sb.append("<br/><font color=red><b><em>Deprecated</em></b></font>");
      }
      else {
        sb.append("<br/><b><em>Deprecated</em></b>");
      }

      final String reasonText = deprecation.getReason().getFullText();
      if (StringUtil.isNotEmpty(reasonText)) {
        sb.append("  ").append(reasonText);
      }
      final String replacement = deprecation.getReplacement();
      if (replacement != null) {
        sb.append("<br/><em>See:</em> ");
        DocumentationManagerUtil.createHyperlink(sb, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX + module.getName() +
                                                     CONFIG_KEY_LINK_SEPARATOR + replacement,
                                                 replacement, true);
      }
      sb.append("<br/>");
    }

    final String defaultValue = key.getDefaultValue();
    if (defaultValue != null) {
      sb.append("<br/>Default: <em>").append(defaultValue).append("</em>");
    }

    sb.append("<br/><br/>");
    final SpringBootApplicationMetaConfigKey.ItemHint itemHint = key.getItemHint();
    final String fullDescription = key.getDescriptionText().getFullText();
    final boolean hasDescription = StringUtil.isNotEmpty(fullDescription);
    if (hasDescription) {
      sb.append(fullDescription);
    }

    final List<SpringBootApplicationMetaConfigKey.ValueHint> valueHints = itemHint.getValueHints();
    appendValueDescriptionTable(sb, valueHints,
                                SpringBootApplicationMetaConfigKey.ValueHint::getValue,
                                hint -> hint.getDescriptionText().getFullText());

    final PsiClass typeClass = PsiTypesUtil.getPsiClass(key.getType());
    if (typeClass != null &&
        typeClass.isEnum()) {
      final List<PsiField> enumConstants = ContainerUtil.findAll(typeClass.getFields(), Conditions.instanceOf(PsiEnumConstant.class));
      appendValueDescriptionTable(sb, enumConstants,
                                  field -> StringUtil.defaultIfEmpty(field.getName(), "<invalid>").toLowerCase(Locale.US),
                                  field -> {
                                    final PsiElement navigationElement = field.getNavigationElement();
                                    if (!(navigationElement instanceof PsiDocCommentOwner)) return "";

                                    final PsiDocComment comment = ((PsiDocCommentOwner)navigationElement).getDocComment();
                                    if (comment == null) {
                                      return "";
                                    }

                                    StringBuilder doc = new StringBuilder();
                                    for (PsiElement element : comment.getDescriptionElements()) {
                                      doc.append(StringUtil.replaceUnicodeEscapeSequences(element.getText()));
                                    }
                                    return doc.toString();
                                  });
    }

    // show original doc (method/field) last to prevent HTML problems
    if (StringUtil.isEmpty(fullDescription)) {
      PsiElement docElement = getOriginalDocumentationElement(key.getDeclaration().getNavigationElement());
      final String originalDoc = getOriginalDocForPsiMember(docElement);
      if (StringUtil.isNotEmpty(originalDoc)) {
        sb.append("<em>Original documentation:</em><br/>");
        sb.append(originalDoc);
      }
    }
    return sb.toString();
  }

  private static <V> void appendValueDescriptionTable(StringBuilder sb,
                                                      List<V> elements,
                                                      Function<V, String> valueFunction,
                                                      Function<V, String> descriptionFunction) {
    if (elements.isEmpty()) return;

    sb.append("<br/><br/>");
    sb.append("<table cellpadding=\"5\">");
    sb.append("<tr>").append("<td><em>Value</em></td><td><em>Description</em></td>").append("</tr>");
    for (V value : elements) {
      sb.append("<tr>");
      sb.append("<td><pre>").append(valueFunction.fun(value)).append("</pre></td>");
      sb.append("<td>").append(descriptionFunction.fun(value)).append("</td>");
      sb.append("</tr>");
    }
    sb.append("</table>");
  }

  @Nullable
  private static String getOriginalDocForPsiMember(PsiElement docElement) {
    if (!(docElement instanceof PsiMember)) {
      return null;
    }

    final DocumentationProvider provider = DocumentationManager.getProviderFromElement(docElement);
    return provider.generateDoc(docElement, docElement);
  }

  /**
   * Fallback to underlying field (if navigationElement is setter/getter and field has javadoc), otherwise return given navigationElement.
   *
   * @param navigationElement Original navigation element.
   * @return Element to get documentation from.
   */
  @NotNull
  private static PsiElement getOriginalDocumentationElement(PsiElement navigationElement) {
    if (navigationElement instanceof PsiMethod) {
      final PsiField field = PropertyUtilBase.findPropertyFieldByMember((PsiMember)navigationElement);
      if (field != null && field.getDocComment() != null) return field;
    }
    return navigationElement;
  }
}
