// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.ide.presentation.Presentation;
import com.intellij.ide.projectView.PresentationData;
import com.intellij.navigation.ItemPresentation;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.*;
import com.intellij.psi.impl.FakePsiElement;
import com.intellij.psi.presentation.java.SymbolPresentationUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.List;

/**
 * Reference to {@link SpringBootApplicationMetaConfigKey}.
 */
public abstract class MetaConfigKeyReference extends PsiReferenceBase.Poly<PsiElement> {

  private final NotNullLazyValue<List<SpringBootApplicationMetaConfigKey>> myKeys;

  /**
   * @param element Underlying element.
   * @param keyText Key text, might be different from element's text.
   */
  protected MetaConfigKeyReference(PsiElement element, final String keyText) {
    super(element);

    myKeys = new AtomicNotNullLazyValue<List<SpringBootApplicationMetaConfigKey>>() {
      @NotNull
      @Override
      protected List<SpringBootApplicationMetaConfigKey> compute() {
        return getAllKeys(keyText);
      }
    };
  }

  /**
   * Gets all matching config keys for given key text.
   *
   * @param keyText Key to resolve.
   * @return Matching keys.
   */
  @NotNull
  protected List<SpringBootApplicationMetaConfigKey> getAllKeys(String keyText) {
    return SpringBootApplicationMetaConfigKeyManager.getInstance().findAllApplicationMetaConfigKeys(getElement(), keyText);
  }

  /**
   * Creates navigation/"resolve" element to this reference.
   *
   * @return Navigation element.
   */
  public FakePsiElement createNavigationElement() {
    return new MyNavigationFakeElement(this);
  }

  /**
   * Returns the resolved config key for the given "key" element.
   *
   * @param keyElement Element to resolve to config key.
   * @return {@code null} if not resolved.
   */
  @Nullable
  public static SpringBootApplicationMetaConfigKey getResolvedMetaConfigKey(@NotNull PsiElement keyElement) {
    for (PsiReference reference : keyElement.getReferences()) {
      if (reference instanceof MetaConfigKeyReference) {
        return ((MetaConfigKeyReference)reference).getResolvedKey();
      }
    }
    return null;
  }

  /**
   * Text to display in UI for this reference.
   * <p/>
   * Should return configuration file representation of "[qualifiedKey:value]" instead of plain reference text.
   *
   * @return Customized display text.
   */
  @NotNull
  public abstract String getReferenceDisplayText();

  @Nullable
  protected SpringBootApplicationMetaConfigKey getResolvedKey() {
    return ContainerUtil.getFirstItem(myKeys.getValue());
  }

  @NotNull
  @Override
  public ResolveResult[] multiResolve(boolean incompleteCode) {
    return PsiElementResolveResult.createResults(ContainerUtil.map(myKeys.getValue(), SpringBootApplicationMetaConfigKey::getDeclaration));
  }

  @Override
  public PsiElement handleElementRename(String newElementName) throws IncorrectOperationException {
    // do *NOT* rename key automatically for now (reference on property setter via SpringBootConfigurationPropertyReferenceSearcher)
    return getElement();
  }


  @Presentation(typeName = "Configuration Key")
  private static class MyNavigationFakeElement extends FakePsiElement {

    private final PsiElement myElement;
    private final MetaConfigKeyReference myReference;

    private MyNavigationFakeElement(MetaConfigKeyReference reference) {
      myReference = reference;
      myElement = reference.getElement();
    }

    @Override
    public String getName() {
      return myReference.getReferenceDisplayText();
    }

    @NotNull
    @Override
    public PsiElement getNavigationElement() {
      return myElement;
    }

    // GotoRelated popup does not use ItemPresentation :-/
    @Nullable
    @Override
    public Icon getIcon(boolean open) {
      return getListItemIcon(myElement);
    }

    @NotNull
    @Override
    public ItemPresentation getPresentation() {
      return new PresentationData(myReference.getReferenceDisplayText(),
                                  "(" + SymbolPresentationUtil.getFilePathPresentation(myElement.getContainingFile()) + ")",
                                  getListItemIcon(myElement),
                                  null);
    }

    @Override
    public PsiElement getParent() {
      return myElement;
    }

    private static Icon getListItemIcon(PsiElement navigationItem) {
      return navigationItem.getContainingFile().getIcon(0);
    }
  }
}
