// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.text.BreakIterator;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Configuration key defined via metadata.
 */
public interface SpringBootApplicationMetaConfigKey {

  @NotNull
  String getName();

  @NotNull
  PsiElement getDeclaration();

  @NotNull
  DeclarationResolveResult getDeclarationResolveResult();

  /**
   * Declared type.
   *
   * @return Type.
   */
  PsiType getType();

  /**
   * Returns value type.
   * <ul>
   * <li>Array, List: component/list value type</li>
   * <li>Map: map value type</li>
   * <li>"plain": declared type {@link #getType()}</li>
   * </ul>
   *
   * @return Value type calculated from {@link #getType()}.
   * @see AccessType
   * @see #getMapKeyType()
   */
  @Nullable
  PsiType getEffectiveValueType();

  boolean isAccessType(AccessType... types);

  /**
   * Map key type if key has access type {@link AccessType#MAP_GROUP}.
   *
   * @return {@code null} for non-Map keys.
   */
  @Nullable
  PsiClass getMapKeyType();

  @NotNull
  DescriptionText getDescriptionText();

  @NotNull
  Deprecation getDeprecation();

  @Nullable
  String getDefaultValue();

  @NotNull
  ItemHint getItemHint();

  @NotNull
  ItemHint getKeyItemHint();

  @NotNull
  MetaConfigKeyPresentation getPresentation();

  boolean matches(@NotNull String keyText);

  boolean matchesPrefix(@NotNull String prefixText);

  enum AccessType {
    NORMAL,
    MAP,
    ENUM_MAP,
    INDEXED;

    /**
     * All "map-like" (Key/Value) access types.
     */
    public static final AccessType[] MAP_GROUP = new AccessType[]{MAP, ENUM_MAP};
  }

  enum DeclarationResolveResult {

    /**
     * Bean property in configuration bean class.
     */
    PROPERTY,

    /**
     * Defined locally in project via {@value com.intellij.spring.boot.SpringBootConfigFileConstants#ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON}.
     */
    ADDITIONAL_JSON,

    /**
     * "Standalone" definition in {@value com.intellij.spring.boot.SpringBootConfigFileConstants#SPRING_CONFIGURATION_METADATA_JSON}.
     */
    JSON,

    /**
     * Same as {@link #JSON} but with unresolved {@value SpringBootMetadataConstants#SOURCE_TYPE} class.
     */
    JSON_UNRESOLVED_SOURCE_TYPE
  }

  interface MetaConfigKeyPresentation {

    @NotNull
    Icon getIcon();

    LookupElementBuilder getLookupElement();

    /**
     * Create lookup element with custom item text (e.g. for special keys).
     *
     * @param lookupString Custom name.
     */
    LookupElementBuilder getLookupElement(String lookupString);

    /**
     * Tunes final config key lookup element.
     * <p/>
     * Sorts deprecated elements last.
     *
     * @param lookupElement Lookup element.
     * @return Final element.
     */
    LookupElement tuneLookupElement(LookupElement lookupElement);
  }


  class DescriptionText {

    public static final DescriptionText NONE = new DescriptionText("");

    private final String myText;

    public DescriptionText(@NotNull String text) {
      myText = text;
    }

    /**
     * Returns full description text.
     *
     * @return Description text or empty String if none defined.
     */
    @NotNull
    public String getFullText() {
      return myText;
    }

    /**
     * Returns short description text, e.g. for use in lookup list.
     *
     * @return First sentence (without ".") or empty String if none defined.
     */
    @NotNull
    public String getShortText() {
      String firstSentence = myText;
      if (StringUtil.containsChar(myText, '.')) {
        final BreakIterator breakIterator = BreakIterator.getSentenceInstance(Locale.US);
        breakIterator.setText(myText);
        firstSentence = myText.substring(breakIterator.first(), breakIterator.next()).trim();
      }
      if (StringUtil.isNotEmpty(firstSentence)) {
        final String withoutDot = StringUtil.endsWithChar(firstSentence, '.') ?
                                  firstSentence.substring(0, firstSentence.length() - 1) : firstSentence;
        return StringUtil.replace(withoutDot, "\n", "");
      }
      return "";
    }

    @Override
    public String toString() {
      return "DescriptionText{" +
             "myText='" + myText + '\'' +
             '}';
    }
  }


  /**
   * SB 1.3
   */
  class Deprecation {

    public enum DeprecationLevel {
      WARNING("warning"),
      ERROR("error");

      private final String myValue;

      DeprecationLevel(String value) {
        myValue = value;
      }

      public String getValue() {
        return myValue;
      }

      @Nullable
      public static DeprecationLevel parse(@Nullable String value) {
        if (value == null) return null;

        for (DeprecationLevel level : values()) {
          if (level.getValue().equals(value)) {
            return level;
          }
        }
        return null;
      }
    }

    public static final Deprecation NOT_DEPRECATED = new Deprecation(DescriptionText.NONE, null) {
      @Override
      public String toString() {
        return "Deprecation.NOT_DEPRECATED";
      }
    };

    /**
     * Simple {@code deprecated} (pre SB 1.3).
     *
     * @since 2016.3
     */
    public static final Deprecation DEPRECATED_WITHOUT_REASON =
      new SpringBootApplicationMetaConfigKey.Deprecation(DescriptionText.NONE, null);

    private final DescriptionText myReason;
    private final DeprecationLevel myLevel;
    private final String myReplacement;

    Deprecation(@NotNull DescriptionText reason,
                @Nullable String replacement) {
      this(reason, DeprecationLevel.WARNING, replacement);
    }

    Deprecation(@NotNull DescriptionText reason,
                @NotNull DeprecationLevel level,
                @Nullable String replacement) {
      myReason = reason;
      myLevel = level;
      myReplacement = replacement;
    }

    @NotNull
    public DescriptionText getReason() {
      return myReason;
    }

    /**
     * @since 2017.2
     */
    @NotNull
    public DeprecationLevel getLevel() {
      return myLevel;
    }

    @Nullable
    public String getReplacement() {
      return myReplacement;
    }

    @Override
    public String toString() {
      return "Deprecation{" +
             "myReason=" + myReason +
             ", myLevel=" + myLevel +
             ", myReplacement='" + myReplacement + '\'' +
             '}';
    }
  }


  class ValueProvider {

    private final String myName;
    private final Map<String, String> myParameters;

    ValueProvider(String name, Map<String, String> parameters) {
      myName = name;
      myParameters = parameters;
    }

    public String getName() {
      return myName;
    }

    /**
     * Parameter name, parameter value text "as is" (stripped quotes for literals).
     *
     * @return All defined parameters.
     */
    public Map<String, String> getParameters() {
      return myParameters;
    }

    @Override
    public String toString() {
      return "ValueProvider{" +
             "myName='" + myName + '\'' +
             ", myParameters=" + myParameters.size() +
             '}';
    }
  }

  class ValueHint {

    private final String myValue;

    private final DescriptionText myDescriptionText;

    ValueHint(@NotNull String value, @NotNull DescriptionText descriptionText) {
      myValue = value;
      myDescriptionText = descriptionText;
    }

    @NotNull
    public String getValue() {
      return myValue;
    }

    @NotNull
    public DescriptionText getDescriptionText() {
      return myDescriptionText;
    }

    @Override
    public String toString() {
      return "ValueHint{" +
             "myValue='" + myValue + '\'' +
             '}';
    }
  }

  class ItemHint {

    public static final ItemHint NONE = new ItemHint(Collections.emptyList(), Collections.emptyList()) {
      @Override
      public String toString() {
        return "ItemHint.NONE";
      }
    };

    public static final String MAP_KEYS_NAME_SUFFIX = ".keys";
    public static final String MAP_VALUES_NAME_SUFFIX = ".values";

    private final List<ValueProvider> myValueProviders;
    private final List<ValueHint> myValueHints;

    public ItemHint(List<ValueProvider> valueProviders, List<ValueHint> valueHints) {
      myValueProviders = valueProviders;
      myValueHints = valueHints;
    }

    public List<ValueHint> getValueHints() {
      return myValueHints;
    }

    public List<ValueProvider> getValueProviders() {
      return myValueProviders;
    }

    @Override
    public String toString() {
      return "ItemHint{" +
             "myValueProviders=" + myValueProviders +
             ", myValueHints=" + myValueHints +
             '}';
    }
  }
}
