// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata.additional;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.ProblemDescriptor;
import com.intellij.icons.AllIcons;
import com.intellij.internal.statistic.UsageTrigger;
import com.intellij.json.psi.*;
import com.intellij.openapi.application.Result;
import com.intellij.openapi.application.WriteAction;
import com.intellij.openapi.command.WriteCommandAction;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectUtil;
import com.intellij.openapi.ui.Messages;
import com.intellij.openapi.ui.popup.JBPopupFactory;
import com.intellij.openapi.util.ThrowableComputable;
import com.intellij.openapi.vfs.ReadonlyStatusHandler;
import com.intellij.openapi.vfs.VfsUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiManager;
import com.intellij.psi.codeStyle.CodeStyleManager;
import com.intellij.spring.boot.SpringBootConfigFileConstants;
import com.intellij.spring.boot.application.metadata.SpringBootMetadataConstants;
import com.intellij.spring.statistics.SpringStatisticsConstants;
import com.intellij.ui.ListCellRendererWrapper;
import com.intellij.ui.components.JBList;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.ObjectUtils;
import com.intellij.util.Processor;
import com.intellij.util.ThrowableRunnable;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

public class DefineLocalMetaConfigKeyFix implements LocalQuickFix {

  private final String myKeyName;

  public DefineLocalMetaConfigKeyFix(String keyName) {
    myKeyName = keyName;
  }

  @Nls
  @NotNull
  @Override
  public String getName() {
    return "Define configuration key '" + myKeyName + "'";
  }

  @Override
  public boolean startInWriteAction() {
    return false;
  }

  @NotNull
  @Override
  public String getFamilyName() {
    return "Define configuration key";
  }

  @Override
  public void applyFix(@NotNull final Project project, @NotNull ProblemDescriptor descriptor) {
    final PsiElement element = descriptor.getPsiElement();
    final Module module = ModuleUtilCore.findModuleForPsiElement(element);
    if (module == null) {
      return;
    }
    final String moduleName = module.getName();


    SpringBootAdditionalConfigUtils additionalConfigUtils = new SpringBootAdditionalConfigUtils(module);
    if (!additionalConfigUtils.hasResourceRoots()) {
      Messages.showWarningDialog(project,
                                 "No resources roots found in module '" + moduleName + "'", "Spring Boot");
      return;
    }

    final Processor<JsonFile> addKeyToExistingProcessor = jsonFile -> {
      addKey(project, jsonFile);
      return false;
    };
    if (!additionalConfigUtils.processAdditionalMetadataFiles(addKeyToExistingProcessor)) {
      return;
    }

    final int createNewFile =
      Messages.showYesNoDialog(project,
                               "Could not locate META-INF/" +
                               SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON +
                               " in any of resources roots in module '" + module.getName() + "'." +
                               "\n\n" +
                               "Do you want to create new file for storing additional meta-data?",
                               "Spring Boot", null);
    if (createNewFile == Messages.NO) {
      return;
    }

    final JBList<VirtualFile> list = new JBList<>(additionalConfigUtils.getResourceRoots());
    list.setCellRenderer(new ListCellRendererWrapper<VirtualFile>() {
      @Override
      public void customize(JList list, VirtualFile value, int index, boolean selected, boolean hasFocus) {
        setText(ProjectUtil.calcRelativeToProjectPath(value, project));
        setIcon(AllIcons.Modules.ResourcesRoot);
      }
    });
    JBPopupFactory.getInstance().createListPopupBuilder(list)
      .setTitle("Select Resources Root")
      .setItemChoosenCallback(() -> {
        final VirtualFile selectedRoot = list.getSelectedValue();
        if (selectedRoot == null) return;

        new WriteCommandAction<Void>(project) {
          @Override
          protected void run(@NotNull Result<Void> result) throws Throwable {
            final VirtualFile metaInf = VfsUtil.createDirectoryIfMissing(selectedRoot, "META-INF");

            final VirtualFile addVf = metaInf.createChildData(project,
                                                              SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON);

            VfsUtil.saveText(addVf, "{ \"" + SpringBootMetadataConstants.PROPERTIES + "\": [ ] }");

            final JsonFile jsonFile = (JsonFile)PsiManager.getInstance(project).findFile(addVf);
            assert jsonFile != null;
            addKey(project, jsonFile);
          }
        }.execute().throwException();
      })
      .createPopup().showCenteredInCurrentWindow(project);
  }

  private void addKey(@NotNull Project project, JsonFile additionalJson) {
    if (!ReadonlyStatusHandler.ensureFilesWritable(project, additionalJson.getVirtualFile())) {
      return;
    }

    JsonElementGenerator generator = new JsonElementGenerator(project);
    JsonArray propertiesArray = findOrCreatePropertiesArray(generator, additionalJson);
    if (propertiesArray == null) {
      Messages.showWarningDialog(project,
                                 "Invalid JSON structure in " + additionalJson.getVirtualFile().getPath(),
                                 "Spring Boot");
      return;
    }

    WriteAction.run((ThrowableRunnable<IncorrectOperationException>)() -> {
      final JsonObject value =
        generator.createValue("{\n" +
                              "  \"" + SpringBootMetadataConstants.NAME + "\": \"" + myKeyName + "\",\n" +
                              "  \"" + SpringBootMetadataConstants.TYPE + "\": \"java.lang.String\",\n" +
                              "  \"" + SpringBootMetadataConstants.DESCRIPTION + "\": \"Description for " + myKeyName + ".\"" +
                              "}");

      boolean hasValues = !propertiesArray.getValueList().isEmpty();
      if (hasValues) {
        propertiesArray.addBefore(generator.createComma(), propertiesArray.getLastChild());
      }

      final JsonObject added = (JsonObject)propertiesArray.addBefore(value, propertiesArray.getLastChild());

      CodeStyleManager.getInstance(project).reformatText(additionalJson,
                                                         0,
                                                         additionalJson.getTextLength());
      added.navigate(true);

      UsageTrigger.trigger(SpringStatisticsConstants.USAGE_TRIGGER_PREFIX + "SpringBoot.DefineLocalMetaConfigKeyFix");
    });
  }

  @Nullable
  private static JsonArray findOrCreatePropertiesArray(JsonElementGenerator generator, JsonFile additionalJson) {
    JsonObject rootObject = ObjectUtils.tryCast(additionalJson.getTopLevelValue(), JsonObject.class);
    if (rootObject == null) return null;

    final JsonProperty propertiesRoot = rootObject.findProperty(SpringBootMetadataConstants.PROPERTIES);
    if (propertiesRoot == null) {
      return WriteAction.compute((ThrowableComputable<JsonArray, IncorrectOperationException>)() -> {
        final JsonProperty propertiesProperty =
          generator.createProperty(SpringBootMetadataConstants.PROPERTIES, "[]");
        if (!rootObject.getPropertyList().isEmpty()) {
          rootObject.addBefore(generator.createComma(), rootObject.getLastChild());
        }

        final JsonProperty propertiesAdded = (JsonProperty)rootObject.addBefore(propertiesProperty, rootObject.getLastChild());
        return (JsonArray)propertiesAdded.getValue();
      });
    }

    return ObjectUtils.tryCast(propertiesRoot.getValue(), JsonArray.class);
  }
}
