// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.library;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.roots.libraries.JarVersionDetectionUtil;
import com.intellij.patterns.PatternCondition;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.ProjectScope;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.boot.SpringBootClassesConstants;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static com.intellij.psi.util.CachedValueProvider.Result;

public class SpringBootLibraryUtil {

  public static final PatternCondition<PsiElement> SB_1_3_OR_HIGHER =
    new PatternCondition<PsiElement>("SB1_3OrHigher") {
      @Override
      public boolean accepts(@NotNull PsiElement element, ProcessingContext context) {
        final Module module = ModuleUtilCore.findModuleForPsiElement(element);
        return isAtLeastVersion(module, SpringBootVersion.VERSION_1_3_0);
      }
    };

  public static final PatternCondition<PsiElement> SB_1_5_OR_HIGHER =
    new PatternCondition<PsiElement>("SB1_5OrHigher") {
      @Override
      public boolean accepts(@NotNull PsiElement element, ProcessingContext context) {
        final Module module = ModuleUtilCore.findModuleForPsiElement(element);
        return isAtLeastVersion(module, SpringBootVersion.VERSION_1_5_0);
      }
    };

  public static final PatternCondition<PsiElement> SB_2_0_OR_HIGHER =
    new PatternCondition<PsiElement>("SB2_0OrHigher") {
      @Override
      public boolean accepts(@NotNull PsiElement element, ProcessingContext context) {
        final Module module = ModuleUtilCore.findModuleForPsiElement(element);
        return isAtLeastVersion(module, SpringBootVersion.VERSION_2_0_0);
      }
    };

  // must be listed in ascending order
  public enum SpringBootVersion {
    ANY(SpringBootClassesConstants.SPRING_APPLICATION),
    VERSION_1_2_0("org.springframework.boot.jta.XADataSourceWrapper"),
    VERSION_1_3_0("org.springframework.boot.context.event.ApplicationReadyEvent"),
    VERSION_1_4_0("org.springframework.boot.diagnostics.FailureAnalysis"),
    VERSION_1_5_0("org.springframework.boot.autoconfigure.AbstractDatabaseInitializer"),
    VERSION_2_0_0("org.springframework.boot.WebApplicationType");

    private final String myDetectionClassFqn;

    SpringBootVersion(String detectionClassFqn) {
      myDetectionClassFqn = detectionClassFqn;
    }

    boolean isAtLeast(SpringBootVersion reference) {
      if (reference == ANY) {
        return true;
      }
      return this.ordinal() >= reference.ordinal();
    }

    public String getDetectionClassFqn() {
      return myDetectionClassFqn;
    }
  }

  public static boolean hasConfigurationMetadataAnnotationProcessor(@NotNull final Module module) {
    final PsiClass processor = SpringCommonUtils.findLibraryClass(module,
                                                                  SpringBootClassesConstants.CONFIGURATION_METADATA_ANNOTATION_PROCESSOR);
    return processor != null;
  }

  public static boolean hasSpringBootLibrary(final Project project) {
    if (project.isDisposed()) return false;

    return CachedValuesManager.getManager(project).getCachedValue(project, () -> {
      DumbService dumbService = DumbService.getInstance(project);
      if (dumbService.isDumb()) {
        return Result.create(false, dumbService.getModificationTracker());
      }

      final boolean foundMarkerClass =
        JavaPsiFacade.getInstance(project).findClass(SpringBootVersion.ANY.getDetectionClassFqn(),
                                                     ProjectScope.getAllScope(project)) != null;
      return Result.createSingleDependency(foundMarkerClass, ProjectRootManager.getInstance(project));
    });
  }

  public static boolean hasSpringBootLibrary(@Nullable final Module module) {
    return isAtLeastVersion(module, SpringBootVersion.ANY);
  }

  public static boolean isAtLeastVersion(@Nullable final Module module,
                                         final SpringBootVersion version) {
    if (module == null || module.isDisposed()) return false;

    if (!hasSpringBootLibrary(module.getProject())) return false;

    final SpringBootVersion cached = getCachedSpringBootVersion(module);
    return cached != null && cached.isAtLeast(version);
  }

  @Nullable
  public static String getVersionFromJar(@NotNull Module module) {
    return JarVersionDetectionUtil.detectJarVersion(SpringBootVersion.ANY.getDetectionClassFqn(), module);
  }

  public static boolean hasDevtools(@Nullable final Module module) {
    return SpringCommonUtils.findLibraryClass(module, "org.springframework.boot.devtools.RemoteSpringApplication") != null;
  }

  public static boolean hasActuators(@Nullable final Module module) {
    if (isAtLeastVersion(module, SpringBootVersion.VERSION_2_0_0)) {
      return SpringCommonUtils.findLibraryClass(module, "org.springframework.boot.actuate.endpoint.annotation.Endpoint") != null;
    }
    return SpringCommonUtils.findLibraryClass(module, "org.springframework.boot.actuate.endpoint.Endpoint") != null;
  }

  public static boolean hasRequestMappings(@Nullable final Module module) {
    return SpringCommonUtils.findLibraryClass(module, "org.springframework.web.servlet.handler.AbstractHandlerMethodMapping") != null;
  }

  @Nullable
  private static SpringBootVersion getCachedSpringBootVersion(@NotNull final Module module) {
    final Project project = module.getProject();

    return CachedValuesManager.getManager(project).getCachedValue(module, () -> {
      final GlobalSearchScope scope = GlobalSearchScope.moduleRuntimeScope(module, false);
      final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(project);

      SpringBootVersion detected = null;
      for (SpringBootVersion version : ArrayUtil.reverseArray(SpringBootVersion.values())) {
        final PsiClass psiClass = javaPsiFacade.findClass(version.getDetectionClassFqn(), scope);
        if (psiClass != null) {
          detected = version;
          break;
        }
      }

      return Result.create(detected, ProjectRootManager.getInstance(project));
    });
  }
}
