// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model

import com.intellij.openapi.module.Module
import com.intellij.openapi.util.Ref
import com.intellij.psi.PsiManager
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKey.Deprecation
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKeyManager
import com.intellij.util.Processor
import com.intellij.util.containers.ContainerUtil

/**
 * Search/process configuration files values.
 *
 * @since 2017.3
 */
class SpringBootConfigValueSearcher(private val myModule: Module, private val myIncludeTests: Boolean = false,
                                    private val myConfigKey: String, private val myCheckRelaxedNames: Boolean = true,
                                    private val myActiveProfiles: Set<String>? = null) {

  fun findValueText(): String? {
    val valueText = Ref.create<String>()
    val findValueTextProcessor = Processor<SpringBootModelConfigFileContributor.ConfigurationValueResult> { result ->
      val text = result.valueText ?: return@Processor true

      valueText.set(text)
      false
    }
    process(findValueTextProcessor)
    return valueText.get()
  }

  fun process(processor: Processor<SpringBootModelConfigFileContributor.ConfigurationValueResult>): Boolean {
    val metaConfigKey = SpringBootApplicationMetaConfigKeyManager.getInstance().findApplicationMetaConfigKey(myModule,
                                                                                                             myConfigKey) ?: return true
    if (metaConfigKey.deprecation.level == Deprecation.DeprecationLevel.ERROR) {
      return true
    }

    val psiManager = PsiManager.getInstance(myModule.project)
    for (contributor in SpringBootModelConfigFileContributor.EP_NAME.extensions) {
      for (virtualFile in contributor.getConfigurationFiles(myModule, myIncludeTests)) {
        val configPsiFile = psiManager.findFile(virtualFile) ?: continue

        val result = contributor.findConfigurationValues(configPsiFile, metaConfigKey, myCheckRelaxedNames, myActiveProfiles)
        if (!ContainerUtil.process(result, processor)) return false
      }
    }

    return true
  }

  companion object {

    @JvmStatic
    fun productionForProfiles(module: Module, configKey: String, activeProfiles: Set<String>?): SpringBootConfigValueSearcher {
      return SpringBootConfigValueSearcher(myModule = module, myConfigKey = configKey, myActiveProfiles = activeProfiles)
    }
  }
}
