// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Key;
import com.intellij.spring.facet.SpringFileSet;
import com.intellij.spring.facet.beans.CustomSetting;
import org.jetbrains.annotations.NotNull;

/**
 * Provides file names (default ({@code application|bootstrap}) and/or custom named)
 * used by {@link SpringBootModelConfigFileContributor} to collect config files.
 *
 * @author Konstantin Aleev
 * @since 2017.3
 */
public abstract class SpringBootModelConfigFileNameContributor {
  public static final ExtensionPointName<SpringBootModelConfigFileNameContributor> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.modelConfigFileNameContributor");

  private final CustomSettingDescriptor myCustomNameSettingDescriptor;
  private final CustomSettingDescriptor myCustomFilesSettingDescriptor;
  private final CustomizationPresentation myCustomizationPresentation;

  protected SpringBootModelConfigFileNameContributor(@NotNull CustomSettingDescriptor customNameSettingDescriptor,
                                                     @NotNull CustomSettingDescriptor customFilesSettingDescriptor,
                                                     @NotNull CustomizationPresentation presentation) {
    myCustomNameSettingDescriptor = customNameSettingDescriptor;
    myCustomFilesSettingDescriptor = customFilesSettingDescriptor;
    myCustomizationPresentation = presentation;
  }

  @NotNull
  public final CustomSettingDescriptor getCustomNameSettingDescriptor() {
    return myCustomNameSettingDescriptor;
  }

  @NotNull
  public final CustomSettingDescriptor getCustomFilesSettingDescriptor() {
    return myCustomFilesSettingDescriptor;
  }

  @NotNull
  public CustomizationPresentation getCustomizationPresentation() {
    return myCustomizationPresentation;
  }

  public abstract boolean accept(SpringFileSet fileSet);

  public abstract boolean accept(Module module);

  public static class CustomSettingDescriptor {
    public final Key<CustomSetting.STRING> key;
    public final String description;
    public final String defaultValue;

    public CustomSettingDescriptor(Key<CustomSetting.STRING> key, String description, String defaultValue) {
      this.key = key;
      this.description = description;
      this.defaultValue = defaultValue;
    }

    public CustomSetting.STRING createCustomSetting() {
      return new CustomSetting.STRING(key, description, defaultValue);
    }
  }

  public static class CustomizationPresentation {
    public final String configFileKey;
    public final String customizationPanelTitle;

    public CustomizationPresentation(String configFileKey, String customizationPanelTitle) {
      this.configFileKey = configFileKey;
      this.customizationPanelTitle = customizationPanelTitle;
    }
  }
}
